"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_compat_1 = require("./construct-compat");
const prepare_app_1 = require("./private/prepare-app");
const runtime_info_1 = require("./private/runtime-info");
const tree_metadata_1 = require("./private/tree-metadata");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends construct_compat_1.Construct {
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_compat_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined,
        });
        this._assembly = assembly;
        return assembly;
    }
    prepare() {
        super.prepare();
        prepare_app_1.prepareApp(this);
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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