"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const chain_1 = require("../chain");
const fields_1 = require("../fields");
const util_1 = require("../util");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 */
class Task extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.timeout = props.timeout;
        const taskProps = props.task.bind(this);
        this.taskProps = {
            ...taskProps,
            parameters: util_1.noEmptyObject({ ...taskProps.parameters || {}, ...props.parameters || {} }),
        };
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderInputOutput(),
            Type: state_type_1.StateType.TASK,
            Comment: this.comment,
            Resource: this.taskProps.resourceArn,
            Parameters: this.taskProps.parameters && fields_1.FieldUtils.renderObject(this.taskProps.parameters),
            ResultPath: state_1.renderJsonPath(this.resultPath),
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
            HeartbeatSeconds: this.taskProps.heartbeat && this.taskProps.heartbeat.toSeconds(),
        };
    }
    /**
     * Return the given named metric for this Task
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: this.taskProps.metricDimensions,
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskProps.policyStatements || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('This Task Resource does not expose metrics');
        }
        return this.metric(prefix + suffix, props);
    }
}
exports.Task = Task;
//# sourceMappingURL=data:application/json;base64,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