"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const JSON_PATH_TOKEN_SYMBOL = Symbol.for('@aws-cdk/aws-stepfunctions.JsonPathToken');
class JsonPathToken {
    constructor(path) {
        this.path = path;
        this.creationStack = core_1.captureStackTrace();
        this.displayHint = path.replace(/^[^a-zA-Z]+/, '');
        Object.defineProperty(this, JSON_PATH_TOKEN_SYMBOL, { value: true });
    }
    static isJsonPathToken(x) {
        return x[JSON_PATH_TOKEN_SYMBOL] === true;
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.JsonPathToken = JsonPathToken;
/**
 * Deep render a JSON object to expand JSON path fields, updating the key to end in '.$'
 */
function renderObject(obj) {
    return recurseObject(obj, {
        handleString: renderString,
        handleList: renderStringList,
        handleNumber: renderNumber,
        handleBoolean: renderBoolean,
    });
}
exports.renderObject = renderObject;
/**
 * Return all JSON paths that are used in the given structure
 */
function findReferencedPaths(obj) {
    const found = new Set();
    recurseObject(obj, {
        handleString(_key, x) {
            const path = jsonPathString(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleList(_key, x) {
            const path = jsonPathStringList(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleNumber(_key, x) {
            const path = jsonPathNumber(x);
            if (path !== undefined) {
                found.add(path);
            }
            return {};
        },
        handleBoolean(_key, _x) {
            return {};
        },
    });
    return found;
}
exports.findReferencedPaths = findReferencedPaths;
function recurseObject(obj, handlers) {
    if (obj === undefined) {
        return undefined;
    }
    const ret = {};
    for (const [key, value] of Object.entries(obj)) {
        if (typeof value === 'string') {
            Object.assign(ret, handlers.handleString(key, value));
        }
        else if (typeof value === 'number') {
            Object.assign(ret, handlers.handleNumber(key, value));
        }
        else if (Array.isArray(value)) {
            Object.assign(ret, recurseArray(key, value, handlers));
        }
        else if (typeof value === 'boolean') {
            Object.assign(ret, handlers.handleBoolean(key, value));
        }
        else if (value === null || value === undefined) {
            // Nothing
        }
        else if (typeof value === 'object') {
            ret[key] = recurseObject(value, handlers);
        }
    }
    return ret;
}
exports.recurseObject = recurseObject;
/**
 * Render an array that may or may not contain a string list token
 */
function recurseArray(key, arr, handlers) {
    if (isStringArray(arr)) {
        const path = jsonPathStringList(arr);
        if (path !== undefined) {
            return handlers.handleList(key, arr);
        }
        // Fall through to correctly reject encoded strings inside an array.
        // They cannot be represented because there is no key to append a '.$' to.
    }
    return {
        [key]: arr.map(value => {
            if ((typeof value === 'string' && jsonPathString(value) !== undefined)
                || (typeof value === 'number' && jsonPathNumber(value) !== undefined)
                || (isStringArray(value) && jsonPathStringList(value) !== undefined)) {
                throw new Error('Cannot use JsonPath fields in an array, they must be used in objects');
            }
            if (typeof value === 'object' && value !== null) {
                return recurseObject(value, handlers);
            }
            return value;
        }),
    };
}
function isStringArray(x) {
    return Array.isArray(x) && x.every(el => typeof el === 'string');
}
/**
 * Render a parameter string
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderString(key, value) {
    const path = jsonPathString(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter string list
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderStringList(key, value) {
    const path = jsonPathStringList(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter number
 *
 * If the string value starts with '$.', render it as a path string, otherwise as a direct string.
 */
function renderNumber(key, value) {
    const path = jsonPathNumber(value);
    if (path !== undefined) {
        return { [key + '.$']: path };
    }
    else {
        return { [key]: value };
    }
}
/**
 * Render a parameter boolean
 */
function renderBoolean(key, value) {
    return { [key]: value };
}
/**
 * If the indicated string is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathString(x) {
    const fragments = core_1.Tokenization.reverseString(x);
    const jsonPathTokens = fragments.tokens.filter(JsonPathToken.isJsonPathToken);
    if (jsonPathTokens.length > 0 && fragments.length > 1) {
        throw new Error(`Field references must be the entire string, cannot concatenate them (found '${x}')`);
    }
    if (jsonPathTokens.length > 0) {
        return jsonPathTokens[0].path;
    }
    return undefined;
}
exports.jsonPathString = jsonPathString;
/**
 * If the indicated string list is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathStringList(x) {
    return pathFromToken(core_1.Tokenization.reverseList(x));
}
/**
 * If the indicated number is an encoded JSON path, return the path
 *
 * Otherwise return undefined.
 */
function jsonPathNumber(x) {
    return pathFromToken(core_1.Tokenization.reverseNumber(x));
}
function pathFromToken(token) {
    return token && (JsonPathToken.isJsonPathToken(token) ? token.path : undefined);
}
//# sourceMappingURL=data:application/json;base64,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