"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data that gets passed around between states
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() {
    }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() {
    }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() {
    }
}
exports.FieldUtils = FieldUtils;
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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