"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Starts an AWS Glue job in a Task state
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class GlueStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.getPolicies();
        this.taskMetrics = {
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.props.glueJobName },
        };
    }
    renderTask() {
        var _a, _b;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        return {
            Resource: task_utils_1.integrationResourceArn('glue', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobName: this.props.glueJobName,
                Arguments: (_a = this.props.arguments) === null || _a === void 0 ? void 0 : _a.value,
                Timeout: (_b = this.props.timeout) === null || _b === void 0 ? void 0 : _b.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            }),
            TimeoutSeconds: undefined,
        };
    }
    getPolicies() {
        let iamActions;
        if (this.integrationPattern === sfn.IntegrationPattern.REQUEST_RESPONSE) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return [new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'glue',
                        resource: 'job',
                        resourceName: this.props.glueJobName,
                    }),
                ],
                actions: iamActions,
            })];
    }
}
exports.GlueStartJobRun = GlueStartJobRun;
GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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