"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A Step Functions Task to create an EMR Cluster.
 *
 * The ClusterConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the ClusterId.
 *
 * @experimental
 */
class EmrCreateCluster {
    constructor(props) {
        this.props = props;
        this.visibleToAllUsers = (this.props.visibleToAllUsers !== undefined) ? this.props.visibleToAllUsers : true;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        this._serviceRole = this.props.serviceRole;
        this._clusterRole = this.props.clusterRole;
        this._autoScalingRole = this.props.autoScalingRole;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call CreateCluster.`);
        }
    }
    /**
     * The service role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get serviceRole() {
        if (this._serviceRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._serviceRole;
    }
    /**
     * The instance role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get clusterRole() {
        if (this._clusterRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._clusterRole;
    }
    /**
     * The autoscaling role for the EMR Cluster.
     *
     * Only available after task has been added to a state machine.
     */
    get autoScalingRole() {
        if (this._autoScalingRole === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._autoScalingRole;
    }
    bind(task) {
        var _a, _b;
        // If the Roles are undefined then they weren't provided, so create them
        this._serviceRole = this._serviceRole || this.createServiceRole(task);
        this._clusterRole = this._clusterRole || this.createClusterRole(task);
        // AutoScaling roles are not valid with InstanceFleet clusters.
        // Attempt to create only if .instances.instanceFleets is undefined or empty
        if (this.props.instances.instanceFleets === undefined || this.props.instances.instanceFleets.length === 0) {
            this._autoScalingRole = this._autoScalingRole || this.createAutoScalingRole(task);
            // If InstanceFleets are used and an AutoScaling Role is specified, throw an error
        }
        else if (this._autoScalingRole !== undefined) {
            throw new Error('Auto Scaling roles can not be specified with instance fleets.');
        }
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('elasticmapreduce', 'createCluster', this.integrationPattern),
            policyStatements: this.createPolicyStatements(task, this._serviceRole, this._clusterRole, this._autoScalingRole),
            parameters: {
                Instances: EmrCreateCluster.InstancesConfigPropertyToJson(this.props.instances),
                JobFlowRole: cdk.stringToCloudFormation(this._clusterRole.roleName),
                Name: cdk.stringToCloudFormation(this.props.name),
                ServiceRole: cdk.stringToCloudFormation(this._serviceRole.roleName),
                AdditionalInfo: cdk.stringToCloudFormation(this.props.additionalInfo),
                Applications: cdk.listMapper(EmrCreateCluster.ApplicationConfigPropertyToJson)(this.props.applications),
                AutoScalingRole: cdk.stringToCloudFormation((_a = this._autoScalingRole) === null || _a === void 0 ? void 0 : _a.roleName),
                BootstrapActions: cdk.listMapper(EmrCreateCluster.BootstrapActionConfigToJson)(this.props.bootstrapActions),
                Configurations: cdk.listMapper(EmrCreateCluster.ConfigurationPropertyToJson)(this.props.configurations),
                CustomAmiId: cdk.stringToCloudFormation(this.props.customAmiId),
                EbsRootVolumeSize: cdk.numberToCloudFormation(this.props.ebsRootVolumeSize),
                KerberosAttributes: (this.props.kerberosAttributes === undefined) ?
                    this.props.kerberosAttributes :
                    EmrCreateCluster.KerberosAttributesPropertyToJson(this.props.kerberosAttributes),
                LogUri: cdk.stringToCloudFormation(this.props.logUri),
                ReleaseLabel: cdk.stringToCloudFormation(this.props.releaseLabel),
                ScaleDownBehavior: cdk.stringToCloudFormation((_b = this.props.scaleDownBehavior) === null || _b === void 0 ? void 0 : _b.valueOf()),
                SecurityConfiguration: cdk.stringToCloudFormation(this.props.securityConfiguration),
                Tags: cdk.listMapper(cdk.cfnTagToCloudFormation)(this.props.tags),
                VisibleToAllUsers: cdk.booleanToCloudFormation(this.visibleToAllUsers),
            },
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call CreateCluster.
     */
    createPolicyStatements(task, serviceRole, clusterRole, autoScalingRole) {
        const stack = cdk.Stack.of(task);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:RunJobFlow',
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows',
                ],
                resources: ['*'],
            }),
        ];
        // Allow the StateMachine to PassRole to Cluster roles
        policyStatements.push(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [
                serviceRole.roleArn,
                clusterRole.roleArn,
            ],
        }));
        if (autoScalingRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [autoScalingRole.roleArn],
            }));
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.SYNC) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRRunJobFlowRule',
                    })],
            }));
        }
        return policyStatements;
    }
    /**
     * Generate the Role used by the EMR Service
     */
    createServiceRole(task) {
        return new iam.Role(task, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceRole'),
            ],
        });
    }
    /**
     * Generate the Role and Instance Profile used by the EC2 instances
     *
     * Data access permissions will need to be updated by the user
     */
    createClusterRole(task) {
        const role = new iam.Role(task, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        new iam.CfnInstanceProfile(task, 'InstanceProfile', {
            roles: [role.roleName],
            instanceProfileName: role.roleName,
        });
        return role;
    }
    /**
     * Generate the Role used to AutoScale the Cluster
     */
    createAutoScalingRole(task) {
        var _a;
        const role = new iam.Role(task, 'AutoScalingRole', {
            assumedBy: new iam.ServicePrincipal('elasticmapreduce.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonElasticMapReduceforAutoScalingRole'),
            ],
        });
        (_a = role.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            principals: [
                new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
            ],
            actions: [
                'sts:AssumeRole',
            ],
        }));
        return role;
    }
}
exports.EmrCreateCluster = EmrCreateCluster;
(function (EmrCreateCluster) {
    /**
     * Valid valus for the Cluster ScaleDownBehavior
     *
     * @experimental
     */
    let EmrClusterScaleDownBehavior;
    (function (EmrClusterScaleDownBehavior) {
        /**
         * Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was
         * submitted. This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_INSTANCE_HOUR"] = "TERMINATE_AT_INSTANCE_HOUR";
        /**
         * Indicates that Amazon EMR blacklists and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the
         * instance-hour boundary.
         */
        EmrClusterScaleDownBehavior["TERMINATE_AT_TASK_COMPLETION"] = "TERMINATE_AT_TASK_COMPLETION";
    })(EmrClusterScaleDownBehavior = EmrCreateCluster.EmrClusterScaleDownBehavior || (EmrCreateCluster.EmrClusterScaleDownBehavior = {}));
    /**
     * Instance Role Types
     *
     * @experimental
     */
    let InstanceRoleType;
    (function (InstanceRoleType) {
        /**
         * Master Node
         */
        InstanceRoleType["MASTER"] = "MASTER";
        /**
         * Core Node
         */
        InstanceRoleType["CORE"] = "CORE";
        /**
         * Task Node
         */
        InstanceRoleType["TASK"] = "TASK";
    })(InstanceRoleType = EmrCreateCluster.InstanceRoleType || (EmrCreateCluster.InstanceRoleType = {}));
    /**
     * EBS Volume Types
     *
     * @experimental
     */
    let EbsBlockDeviceVolumeType;
    (function (EbsBlockDeviceVolumeType) {
        /**
         * gp2 Volume Type
         */
        EbsBlockDeviceVolumeType["GP2"] = "gp2";
        /**
         * io1 Volume Type
         */
        EbsBlockDeviceVolumeType["IO1"] = "io1";
        /**
         * Standard Volume Type
         */
        EbsBlockDeviceVolumeType["STANDARD"] = "standard";
    })(EbsBlockDeviceVolumeType = EmrCreateCluster.EbsBlockDeviceVolumeType || (EmrCreateCluster.EbsBlockDeviceVolumeType = {}));
    /**
     * Render the EbsBlockDeviceConfigProperty as JSON
     *
     * @param property
     */
    function EbsBlockDeviceConfigPropertyToJson(property) {
        var _a;
        return {
            VolumeSpecification: {
                Iops: cdk.numberToCloudFormation(property.volumeSpecification.iops),
                SizeInGB: cdk.numberToCloudFormation(property.volumeSpecification.sizeInGB),
                VolumeType: cdk.stringToCloudFormation((_a = property.volumeSpecification.volumeType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            },
            VolumesPerInstance: cdk.numberToCloudFormation(property.volumesPerInstance),
        };
    }
    EmrCreateCluster.EbsBlockDeviceConfigPropertyToJson = EbsBlockDeviceConfigPropertyToJson;
    /**
     * Render the EbsConfigurationProperty to JSON
     *
     * @param property
     */
    function EbsConfigurationPropertyToJson(property) {
        return {
            EbsBlockDeviceConfigs: cdk.listMapper(EbsBlockDeviceConfigPropertyToJson)(property.ebsBlockDeviceConfigs),
            EbsOptimized: cdk.booleanToCloudFormation(property.ebsOptimized),
        };
    }
    EmrCreateCluster.EbsConfigurationPropertyToJson = EbsConfigurationPropertyToJson;
    /**
     * Render the InstanceTypeConfigProperty to JSON]
     *
     * @param property
     */
    function InstanceTypeConfigPropertyToJson(property) {
        var _a;
        return {
            BidPrice: cdk.stringToCloudFormation(property.bidPrice),
            BidPriceAsPercentageOfOnDemandPrice: cdk.numberToCloudFormation(property.bidPriceAsPercentageOfOnDemandPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceType: cdk.stringToCloudFormation((_a = property.instanceType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            WeightedCapacity: cdk.numberToCloudFormation(property.weightedCapacity),
        };
    }
    EmrCreateCluster.InstanceTypeConfigPropertyToJson = InstanceTypeConfigPropertyToJson;
    /**
     * Spot Timeout Actions
     *
     * @experimental
     */
    let SpotTimeoutAction;
    (function (SpotTimeoutAction) {
        /**\
         * SWITCH_TO_ON_DEMAND
         */
        SpotTimeoutAction["SWITCH_TO_ON_DEMAND"] = "SWITCH_TO_ON_DEMAND";
        /**
         * TERMINATE_CLUSTER
         */
        SpotTimeoutAction["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    })(SpotTimeoutAction = EmrCreateCluster.SpotTimeoutAction || (EmrCreateCluster.SpotTimeoutAction = {}));
    /**
     * Render the InstanceFleetProvisioningSpecificationsProperty to JSON
     *
     * @param property
     */
    function InstanceFleetProvisioningSpecificationsPropertyToJson(property) {
        var _a;
        return {
            SpotSpecification: {
                BlockDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.blockDurationMinutes),
                TimeoutAction: cdk.stringToCloudFormation((_a = property.spotSpecification.timeoutAction) === null || _a === void 0 ? void 0 : _a.valueOf()),
                TimeoutDurationMinutes: cdk.numberToCloudFormation(property.spotSpecification.timeoutDurationMinutes),
            },
        };
    }
    EmrCreateCluster.InstanceFleetProvisioningSpecificationsPropertyToJson = InstanceFleetProvisioningSpecificationsPropertyToJson;
    /**
     * Render the InstanceFleetConfigProperty as JSON
     *
     * @param property
     */
    function InstanceFleetConfigPropertyToJson(property) {
        var _a;
        return {
            InstanceFleetType: cdk.stringToCloudFormation((_a = property.instanceFleetType) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceTypeConfigs: cdk.listMapper(InstanceTypeConfigPropertyToJson)(property.instanceTypeConfigs),
            LaunchSpecifications: (property.launchSpecifications === undefined) ?
                property.launchSpecifications :
                InstanceFleetProvisioningSpecificationsPropertyToJson(property.launchSpecifications),
            Name: cdk.stringToCloudFormation(property.name),
            TargetOnDemandCapacity: cdk.numberToCloudFormation(property.targetOnDemandCapacity),
            TargetSpotCapacity: cdk.numberToCloudFormation(property.targetSpotCapacity),
        };
    }
    EmrCreateCluster.InstanceFleetConfigPropertyToJson = InstanceFleetConfigPropertyToJson;
    /**
     * CloudWatch Alarm Comparison Operators
     *
     * @experimental
     */
    let CloudWatchAlarmComparisonOperator;
    (function (CloudWatchAlarmComparisonOperator) {
        /**
         * GREATER_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN_OR_EQUAL"] = "GREATER_THAN_OR_EQUAL";
        /**
         * GREATER_THAN
         */
        CloudWatchAlarmComparisonOperator["GREATER_THAN"] = "GREATER_THAN";
        /**
         * LESS_THAN
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN"] = "LESS_THAN";
        /**
         * LESS_THAN_OR_EQUAL
         */
        CloudWatchAlarmComparisonOperator["LESS_THAN_OR_EQUAL"] = "LESS_THAN_OR_EQUAL";
    })(CloudWatchAlarmComparisonOperator = EmrCreateCluster.CloudWatchAlarmComparisonOperator || (EmrCreateCluster.CloudWatchAlarmComparisonOperator = {}));
    /**
     * CloudWatch Alarm Statistics
     *
     * @experimental
     */
    let CloudWatchAlarmStatistic;
    (function (CloudWatchAlarmStatistic) {
        /**
         * SAMPLE_COUNT
         */
        CloudWatchAlarmStatistic["SAMPLE_COUNT"] = "SAMPLE_COUNT";
        /**
         * AVERAGE
         */
        CloudWatchAlarmStatistic["AVERAGE"] = "AVERAGE";
        /**
         * SUM
         */
        CloudWatchAlarmStatistic["SUM"] = "SUM";
        /**
         * MINIMUM
         */
        CloudWatchAlarmStatistic["MINIMUM"] = "MINIMUM";
        /**
         * MAXIMUM
         */
        CloudWatchAlarmStatistic["MAXIMUM"] = "MAXIMUM";
    })(CloudWatchAlarmStatistic = EmrCreateCluster.CloudWatchAlarmStatistic || (EmrCreateCluster.CloudWatchAlarmStatistic = {}));
    /**
     * CloudWatch Alarm Units
     *
     * @experimental
     */
    let CloudWatchAlarmUnit;
    (function (CloudWatchAlarmUnit) {
        /**
         * NONE
         */
        CloudWatchAlarmUnit["NONE"] = "NONE";
        /**
         * SECONDS
         */
        CloudWatchAlarmUnit["SECONDS"] = "SECONDS";
        /**
         * MICRO_SECONDS
         */
        CloudWatchAlarmUnit["MICRO_SECONDS"] = "MICRO_SECONDS";
        /**
         * MILLI_SECONDS
         */
        CloudWatchAlarmUnit["MILLI_SECONDS"] = "MILLI_SECONDS";
        /**
         * BYTES
         */
        CloudWatchAlarmUnit["BYTES"] = "BYTES";
        /**
         * KILO_BYTES
         */
        CloudWatchAlarmUnit["KILO_BYTES"] = "KILO_BYTES";
        /**
         * MEGA_BYTES
         */
        CloudWatchAlarmUnit["MEGA_BYTES"] = "MEGA_BYTES";
        /**
         * GIGA_BYTES
         */
        CloudWatchAlarmUnit["GIGA_BYTES"] = "GIGA_BYTES";
        /**
         * TERA_BYTES
         */
        CloudWatchAlarmUnit["TERA_BYTES"] = "TERA_BYTES";
        /**
         * BITS
         */
        CloudWatchAlarmUnit["BITS"] = "BITS";
        /**
         * KILO_BITS
         */
        CloudWatchAlarmUnit["KILO_BITS"] = "KILO_BITS";
        /**
         * MEGA_BITS
         */
        CloudWatchAlarmUnit["MEGA_BITS"] = "MEGA_BITS";
        /**
         * GIGA_BITS
         */
        CloudWatchAlarmUnit["GIGA_BITS"] = "GIGA_BITS";
        /**
         * TERA_BITS
         */
        CloudWatchAlarmUnit["TERA_BITS"] = "TERA_BITS";
        /**
         * PERCENT
         */
        CloudWatchAlarmUnit["PERCENT"] = "PERCENT";
        /**
         * COUNT
         */
        CloudWatchAlarmUnit["COUNT"] = "COUNT";
        /**
         * BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["BYTES_PER_SECOND"] = "BYTES_PER_SECOND";
        /**
         * KILO_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BYTES_PER_SECOND"] = "KILO_BYTES_PER_SECOND";
        /**
         * MEGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BYTES_PER_SECOND"] = "MEGA_BYTES_PER_SECOND";
        /**
         * GIGA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BYTES_PER_SECOND"] = "GIGA_BYTES_PER_SECOND";
        /**
         * TERA_BYTES_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BYTES_PER_SECOND"] = "TERA_BYTES_PER_SECOND";
        /**
         * BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["BITS_PER_SECOND"] = "BITS_PER_SECOND";
        /**
         * KILO_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["KILO_BITS_PER_SECOND"] = "KILO_BITS_PER_SECOND";
        /**
         * MEGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["MEGA_BITS_PER_SECOND"] = "MEGA_BITS_PER_SECOND";
        /**
         * GIGA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["GIGA_BITS_PER_SECOND"] = "GIGA_BITS_PER_SECOND";
        /**
         * TERA_BITS_PER_SECOND
         */
        CloudWatchAlarmUnit["TERA_BITS_PER_SECOND"] = "TERA_BITS_PER_SECOND";
        /**
         * COUNT_PER_SECOND
         */
        CloudWatchAlarmUnit["COUNT_PER_SECOND"] = "COUNT_PER_SECOND";
    })(CloudWatchAlarmUnit = EmrCreateCluster.CloudWatchAlarmUnit || (EmrCreateCluster.CloudWatchAlarmUnit = {}));
    /**
     * Render the MetricDimensionProperty as JSON
     *
     * @param property
     */
    function MetricDimensionPropertyToJson(property) {
        return {
            Key: cdk.stringToCloudFormation(property.key),
            Value: cdk.stringToCloudFormation(property.value),
        };
    }
    EmrCreateCluster.MetricDimensionPropertyToJson = MetricDimensionPropertyToJson;
    /**
     * Render the ScalingTriggerProperty to JSON
     *
     * @param property
     */
    function ScalingTriggerPropertyToJson(property) {
        var _a, _b, _c;
        return {
            CloudWatchAlarmDefinition: {
                ComparisonOperator: cdk.stringToCloudFormation((_a = property.cloudWatchAlarmDefinition.comparisonOperator) === null || _a === void 0 ? void 0 : _a.valueOf()),
                Dimensions: cdk.listMapper(MetricDimensionPropertyToJson)(property.cloudWatchAlarmDefinition.dimensions),
                EvaluationPeriods: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.evalutionPeriods),
                MetricName: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.metricName),
                Namespace: cdk.stringToCloudFormation(property.cloudWatchAlarmDefinition.namespace),
                Period: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.period.toSeconds()),
                Statistic: cdk.stringToCloudFormation((_b = property.cloudWatchAlarmDefinition.statistic) === null || _b === void 0 ? void 0 : _b.valueOf()),
                Threshold: cdk.numberToCloudFormation(property.cloudWatchAlarmDefinition.threshold),
                Unit: cdk.stringToCloudFormation((_c = property.cloudWatchAlarmDefinition.unit) === null || _c === void 0 ? void 0 : _c.valueOf()),
            },
        };
    }
    EmrCreateCluster.ScalingTriggerPropertyToJson = ScalingTriggerPropertyToJson;
    /**
     * EC2 Instance Market
     *
     * @experimental
     */
    let InstanceMarket;
    (function (InstanceMarket) {
        /**
         * On Demand Instance
         */
        InstanceMarket["ON_DEMAND"] = "ON_DEMAND";
        /**
         * Spot Instance
         */
        InstanceMarket["SPOT"] = "SPOT";
    })(InstanceMarket = EmrCreateCluster.InstanceMarket || (EmrCreateCluster.InstanceMarket = {}));
    /**
     * AutoScaling Adjustment Type
     *
     * @experimental
     */
    let ScalingAdjustmentType;
    (function (ScalingAdjustmentType) {
        /**
         * CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["CHANGE_IN_CAPACITY"] = "CHANGE_IN_CAPACITY";
        /**
         * PERCENT_CHANGE_IN_CAPACITY
         */
        ScalingAdjustmentType["PERCENT_CHANGE_IN_CAPACITY"] = "PERCENT_CHANGE_IN_CAPACITY";
        /**
         * EXACT_CAPACITY
         */
        ScalingAdjustmentType["EXACT_CAPACITY"] = "EXACT_CAPACITY";
    })(ScalingAdjustmentType = EmrCreateCluster.ScalingAdjustmentType || (EmrCreateCluster.ScalingAdjustmentType = {}));
    /**
     * Render the ScalingActionPropety to JSON
     *
     * @param property
     */
    function ScalingActionPropertyToJson(property) {
        var _a;
        return {
            Market: cdk.stringToCloudFormation((_a = property.market) === null || _a === void 0 ? void 0 : _a.valueOf()),
            SimpleScalingPolicyConfiguration: {
                AdjustmentType: cdk.stringToCloudFormation(property.simpleScalingPolicyConfiguration.adjustmentType),
                CoolDown: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.coolDown),
                ScalingAdjustment: cdk.numberToCloudFormation(property.simpleScalingPolicyConfiguration.scalingAdjustment),
            },
        };
    }
    EmrCreateCluster.ScalingActionPropertyToJson = ScalingActionPropertyToJson;
    /**
     * Render the ScalingRuleProperty to JSON
     *
     * @param property
     */
    function ScalingRulePropertyToJson(property) {
        return {
            Action: ScalingActionPropertyToJson(property.action),
            Description: cdk.stringToCloudFormation(property.description),
            Name: cdk.stringToCloudFormation(property.name),
            Trigger: ScalingTriggerPropertyToJson(property.trigger),
        };
    }
    EmrCreateCluster.ScalingRulePropertyToJson = ScalingRulePropertyToJson;
    /**
     * Render the AutoScalingPolicyProperty to JSON
     *
     * @param property
     */
    function AutoScalingPolicyPropertyToJson(property) {
        return {
            Constraints: {
                MaxCapacity: cdk.numberToCloudFormation(property.constraints.maxCapacity),
                MinCapacity: cdk.numberToCloudFormation(property.constraints.minCapacity),
            },
            Rules: cdk.listMapper(ScalingRulePropertyToJson)(property.rules),
        };
    }
    EmrCreateCluster.AutoScalingPolicyPropertyToJson = AutoScalingPolicyPropertyToJson;
    /**
     * Render the InstanceGroupConfigProperty to JSON
     *
     * @param property
     */
    function InstanceGroupConfigPropertyToJson(property) {
        var _a, _b;
        return {
            AutoScalingPolicy: (property.autoScalingPolicy === undefined) ?
                property.autoScalingPolicy :
                AutoScalingPolicyPropertyToJson(property.autoScalingPolicy),
            BidPrice: cdk.numberToCloudFormation(property.bidPrice),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
            EbsConfiguration: (property.ebsConfiguration === undefined) ?
                property.ebsConfiguration :
                EbsConfigurationPropertyToJson(property.ebsConfiguration),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceRole: cdk.stringToCloudFormation((_a = property.instanceRole) === null || _a === void 0 ? void 0 : _a.valueOf()),
            InstanceType: cdk.stringToCloudFormation(property.instanceType),
            Market: cdk.stringToCloudFormation((_b = property.market) === null || _b === void 0 ? void 0 : _b.valueOf()),
            Name: cdk.stringToCloudFormation(property.name),
        };
    }
    EmrCreateCluster.InstanceGroupConfigPropertyToJson = InstanceGroupConfigPropertyToJson;
    /**
     * Render the PlacementTypeProperty to JSON
     *
     * @param property
     */
    function PlacementTypePropertyToJson(property) {
        return {
            AvailabilityZone: cdk.stringToCloudFormation(property.availabilityZone),
            AvailabilityZones: cdk.listMapper(cdk.stringToCloudFormation)(property.availabilityZones),
        };
    }
    EmrCreateCluster.PlacementTypePropertyToJson = PlacementTypePropertyToJson;
    /**
     * Render the InstancesConfigProperty to JSON
     *
     * @param property
     */
    function InstancesConfigPropertyToJson(property) {
        return {
            AdditionalMasterSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalMasterSecurityGroups),
            AdditionalSlaveSecurityGroups: cdk.listMapper(cdk.stringToCloudFormation)(property.additionalSlaveSecurityGroups),
            Ec2KeyName: cdk.stringToCloudFormation(property.ec2KeyName),
            Ec2SubnetId: cdk.stringToCloudFormation(property.ec2SubnetId),
            Ec2SubnetIds: cdk.listMapper(cdk.stringToCloudFormation)(property.ec2SubnetIds),
            EmrManagedMasterSecurityGroup: cdk.stringToCloudFormation(property.emrManagedMasterSecurityGroup),
            EmrManagedSlaveSecurityGroup: cdk.stringToCloudFormation(property.emrManagedSlaveSecurityGroup),
            HadoopVersion: cdk.stringToCloudFormation(property.hadoopVersion),
            InstanceCount: cdk.numberToCloudFormation(property.instanceCount),
            InstanceFleets: cdk.listMapper(InstanceFleetConfigPropertyToJson)(property.instanceFleets),
            InstanceGroups: cdk.listMapper(InstanceGroupConfigPropertyToJson)(property.instanceGroups),
            KeepJobFlowAliveWhenNoSteps: true,
            MasterInstanceType: cdk.stringToCloudFormation(property.masterInstanceType),
            Placement: (property.placement === undefined) ?
                property.placement :
                PlacementTypePropertyToJson(property.placement),
            ServiceAccessSecurityGroup: cdk.stringToCloudFormation(property.serviceAccessSecurityGroup),
            SlaveInstanceType: cdk.stringToCloudFormation(property.slaveInstanceType),
            TerminationProtected: cdk.booleanToCloudFormation(property.terminationProtected),
        };
    }
    EmrCreateCluster.InstancesConfigPropertyToJson = InstancesConfigPropertyToJson;
    /**
     * Render the ApplicationConfigProperty as JSON
     *
     * @param property
     */
    function ApplicationConfigPropertyToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            Args: cdk.listMapper(cdk.stringToCloudFormation)(property.args),
            Version: cdk.stringToCloudFormation(property.version),
            AdditionalInfo: cdk.objectToCloudFormation(property.additionalInfo),
        };
    }
    EmrCreateCluster.ApplicationConfigPropertyToJson = ApplicationConfigPropertyToJson;
    /**
     * Render the BootstrapActionProperty as JSON
     *
     * @param property
     */
    function BootstrapActionConfigToJson(property) {
        return {
            Name: cdk.stringToCloudFormation(property.name),
            ScriptBootstrapAction: {
                Path: cdk.stringToCloudFormation(property.scriptBootstrapAction.path),
                Args: cdk.listMapper(cdk.stringToCloudFormation)(property.scriptBootstrapAction.args),
            },
        };
    }
    EmrCreateCluster.BootstrapActionConfigToJson = BootstrapActionConfigToJson;
    /**
     * Render the ConfigurationProperty as JSON
     *
     * @param property
     */
    function ConfigurationPropertyToJson(property) {
        return {
            Classification: cdk.stringToCloudFormation(property.classification),
            Properties: cdk.objectToCloudFormation(property.properties),
            Configurations: cdk.listMapper(ConfigurationPropertyToJson)(property.configurations),
        };
    }
    EmrCreateCluster.ConfigurationPropertyToJson = ConfigurationPropertyToJson;
    /**
     * Render the KerberosAttributesProperty as JSON
     *
     * @param property
     */
    function KerberosAttributesPropertyToJson(property) {
        return {
            ADDomainJoinPassword: cdk.stringToCloudFormation(property.adDomainJoinPassword),
            ADDomainJoinUser: cdk.stringToCloudFormation(property.adDomainJoinUser),
            CrossRealmTrustPrincipalPassword: cdk.stringToCloudFormation(property.crossRealmTrustPrincipalPassword),
            KdcAdminPassword: cdk.stringToCloudFormation(property.kdcAdminPassword),
            Realm: cdk.stringToCloudFormation(property.realm),
        };
    }
    EmrCreateCluster.KerberosAttributesPropertyToJson = KerberosAttributesPropertyToJson;
})(EmrCreateCluster = exports.EmrCreateCluster || (exports.EmrCreateCluster = {}));
//# sourceMappingURL=data:application/json;base64,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