import * as sfn from "../../../aws-stepfunctions";
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
export declare enum DynamoConsumedCapacity {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    INDEXES = "INDEXES",
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    TOTAL = "TOTAL",
    /**
     * No ConsumedCapacity details are included in the response.
     */
    NONE = "NONE"
}
/**
 * Determines whether item collection metrics are returned.
 */
export declare enum DynamoItemCollectionMetrics {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    SIZE = "SIZE",
    /**
     * If set to NONE, no statistics are returned.
     */
    NONE = "NONE"
}
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
export declare enum DynamoReturnValues {
    /**
     * Nothing is returned
     */
    NONE = "NONE",
    /**
     * Returns all of the attributes of the item
     */
    ALL_OLD = "ALL_OLD",
    /**
     * Returns only the updated attributes
     */
    UPDATED_OLD = "UPDATED_OLD",
    /**
     * Returns all of the attributes of the item
     */
    ALL_NEW = "ALL_NEW",
    /**
     * Returns only the updated attributes
     */
    UPDATED_NEW = "UPDATED_NEW"
}
/**
 * Map of string to AttributeValue
 */
export interface DynamoAttributeValueMap {
    [key: string]: DynamoAttributeValue;
}
/**
 * Class to generate AttributeValue
 */
export declare class DynamoAttributeValue {
    private attributeValue;
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     */
    withS(value: string): this;
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withN(value: string): this;
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     */
    withB(value: string): this;
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    withSS(value: string[]): this;
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withNS(value: string[]): this;
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    withBS(value: string[]): this;
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    withM(value: DynamoAttributeValueMap): this;
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    withL(value: DynamoAttributeValue[]): this;
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    withNULL(value: boolean): this;
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    withBOOL(value: boolean): this;
    /**
     * Return the attributeValue object
     */
    toObject(): any;
}
/**
 * Property for any key
 */
export interface DynamoAttribute {
    /**
     * The name of the attribute
     */
    readonly name: string;
    /**
     * The value of the attribute
     */
    readonly value: DynamoAttributeValue;
}
/**
 * Class to generate projection expression
 */
export declare class DynamoProjectionExpression {
    private expression;
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr: string): DynamoProjectionExpression;
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index: number): DynamoProjectionExpression;
    /**
     * converts and return the string expression
     */
    toString(): string;
}
/**
 * Properties for DynamoGetItem Task
 */
export interface DynamoGetItemProps {
    /**
     * A attribute representing the partition key of the item to retrieve.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
     */
    readonly partitionKey: DynamoAttribute;
    /**
     * The name of the table containing the requested item.
     */
    readonly tableName: string;
    /**
     * A attribute representing the sort key of the item to retrieve.
     *
     * @default - No sort key
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
     */
    readonly sortKey?: DynamoAttribute;
    /**
     * Determines the read consistency model:
     * If set to true, then the operation uses strongly consistent reads;
     * otherwise, the operation uses eventually consistent reads.
     *
     * @default false
     */
    readonly consistentRead?: boolean;
    /**
     * One or more substitution tokens for attribute names in an expression
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ExpressionAttributeNames
     *
     * @default - No expression attributes
     */
    readonly expressionAttributeNames?: {
        [key: string]: string;
    };
    /**
     * An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table.
     * These attributes can include scalars, sets, or elements of a JSON document.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ProjectionExpression
     *
     * @default - No projection expression
     */
    readonly projectionExpression?: DynamoProjectionExpression[];
    /**
     * Determines the level of detail about provisioned throughput consumption that is returned in the response
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ReturnConsumedCapacity
     *
     * @default DynamoConsumedCapacity.NONE
     */
    readonly returnConsumedCapacity?: DynamoConsumedCapacity;
}
/**
 * Properties for DynamoPutItem Task
 */
export interface DynamoPutItemProps {
    /**
     * A map of attribute name/value pairs, one for each attribute.
     * Only the primary key attributes are required;
     * you can optionally provide other attribute name-value pairs for the item.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-Item
     */
    readonly item: DynamoAttributeValueMap;
    /**
     * The name of the table where the item should be writen .
     */
    readonly tableName: string;
    /**
     * A condition that must be satisfied in order for a conditional PutItem operation to succeed.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ConditionExpression
     *
     * @default - No condition expression
     */
    readonly conditionExpression?: string;
    /**
     * One or more substitution tokens for attribute names in an expression
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeNames
     *
     * @default - No expression attribute names
     */
    readonly expressionAttributeNames?: {
        [key: string]: string;
    };
    /**
     * One or more values that can be substituted in an expression.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeValues
     *
     * @default - No expression attribute values
     */
    readonly expressionAttributeValues?: DynamoAttributeValueMap;
    /**
     * Determines the level of detail about provisioned throughput consumption that is returned in the response
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnConsumedCapacity
     *
     * @default DynamoConsumedCapacity.NONE
     */
    readonly returnConsumedCapacity?: DynamoConsumedCapacity;
    /**
     * The item collection metrics to returned in the response
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LSI.html#LSI.ItemCollections
     *
     * @default DynamoItemCollectionMetrics.NONE
     */
    readonly returnItemCollectionMetrics?: DynamoItemCollectionMetrics;
    /**
     * Use ReturnValues if you want to get the item attributes as they appeared
     * before they were updated with the PutItem request.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnValues
     *
     * @default DynamoReturnValues.NONE
     */
    readonly returnValues?: DynamoReturnValues;
}
/**
 * Properties for DynamoDeleteItem Task
 */
export interface DynamoDeleteItemProps {
    /**
     * An attribute representing the partition key of the item to delete.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-Key
     */
    readonly partitionKey: DynamoAttribute;
    /**
     * The name of the table containing the requested item.
     */
    readonly tableName: string;
    /**
     * An attribute representing the sort key of the item to delete.
     *
     * @default - No sort key
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-Key
     */
    readonly sortKey?: DynamoAttribute;
    /**
     * A condition that must be satisfied in order for a conditional DeleteItem to succeed.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ConditionExpression
     *
     * @default - No condition expression
     */
    readonly conditionExpression?: string;
    /**
     * One or more substitution tokens for attribute names in an expression
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeNames
     *
     * @default - No expression attribute names
     */
    readonly expressionAttributeNames?: {
        [key: string]: string;
    };
    /**
     * One or more values that can be substituted in an expression.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeValues
     *
     * @default - No expression attribute values
     */
    readonly expressionAttributeValues?: DynamoAttributeValueMap;
    /**
     * Determines the level of detail about provisioned throughput consumption that is returned in the response
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnConsumedCapacity
     *
     * @default DynamoConsumedCapacity.NONE
     */
    readonly returnConsumedCapacity?: DynamoConsumedCapacity;
    /**
     * Determines whether item collection metrics are returned.
     * If set to SIZE, the response includes statistics about item collections, if any,
     * that were modified during the operation are returned in the response.
     * If set to NONE (the default), no statistics are returned.
     *
     * @default DynamoItemCollectionMetrics.NONE
     */
    readonly returnItemCollectionMetrics?: DynamoItemCollectionMetrics;
    /**
     * Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnValues
     *
     * @default DynamoReturnValues.NONE
     */
    readonly returnValues?: DynamoReturnValues;
}
/**
 * Properties for DynamoUpdateItem Task
 */
export interface DynamoUpdateItemProps {
    /**
     * The partition key of the item to be updated.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-Key
     */
    readonly partitionKey: DynamoAttribute;
    /**
     * The name of the table containing the requested item.
     */
    readonly tableName: string;
    /**
     * The sort key of the item to be updated.
     *
     * @default - No sort key
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-Key
     */
    readonly sortKey?: DynamoAttribute;
    /**
     * A condition that must be satisfied in order for a conditional DeleteItem to succeed.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ConditionExpression
     *
     * @default - No condition expression
     */
    readonly conditionExpression?: string;
    /**
     * One or more substitution tokens for attribute names in an expression
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeNames
     *
     * @default - No expression attribute names
     */
    readonly expressionAttributeNames?: {
        [key: string]: string;
    };
    /**
     * One or more values that can be substituted in an expression.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeValues
     *
     * @default - No expression attribute values
     */
    readonly expressionAttributeValues?: DynamoAttributeValueMap;
    /**
     * Determines the level of detail about provisioned throughput consumption that is returned in the response
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnConsumedCapacity
     *
     * @default DynamoConsumedCapacity.NONE
     */
    readonly returnConsumedCapacity?: DynamoConsumedCapacity;
    /**
     * Determines whether item collection metrics are returned.
     * If set to SIZE, the response includes statistics about item collections, if any,
     * that were modified during the operation are returned in the response.
     * If set to NONE (the default), no statistics are returned.
     *
     * @default DynamoItemCollectionMetrics.NONE
     */
    readonly returnItemCollectionMetrics?: DynamoItemCollectionMetrics;
    /**
     * Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnValues
     *
     * @default DynamoReturnValues.NONE
     */
    readonly returnValues?: DynamoReturnValues;
    /**
     * An expression that defines one or more attributes to be updated,
     * the action to be performed on them, and new values for them.
     *
     * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-UpdateExpression
     *
     * @default - No update expression
     */
    readonly updateExpression?: string;
}
/**
 * A StepFunctions task to call DynamoGetItem
 */
export declare class DynamoGetItem implements sfn.IStepFunctionsTask {
    private readonly props;
    constructor(props: DynamoGetItemProps);
    bind(_task: sfn.Task): sfn.StepFunctionsTaskConfig;
    private configureProjectionExpression;
}
/**
 * A StepFunctions task to call DynamoPutItem
 */
export declare class DynamoPutItem implements sfn.IStepFunctionsTask {
    private readonly props;
    constructor(props: DynamoPutItemProps);
    bind(_task: sfn.Task): sfn.StepFunctionsTaskConfig;
}
/**
 * A StepFunctions task to call DynamoDeleteItem
 */
export declare class DynamoDeleteItem implements sfn.IStepFunctionsTask {
    private readonly props;
    constructor(props: DynamoDeleteItemProps);
    bind(_task: sfn.Task): sfn.StepFunctionsTaskConfig;
}
/**
 * A StepFunctions task to call DynamoUpdateItem
 */
export declare class DynamoUpdateItem implements sfn.IStepFunctionsTask {
    private readonly props;
    constructor(props: DynamoUpdateItemProps);
    bind(_task: sfn.Task): sfn.StepFunctionsTaskConfig;
}
/**
 * A helper wrapper class to call all DynamoDB APIs
 */
export declare class CallDynamoDB {
    /**
     * Method to get DynamoGetItem task
     *
     * @param props DynamoGetItemProps
     */
    static getItem(props: DynamoGetItemProps): DynamoGetItem;
    /**
     * Method to get DynamoPutItem task
     *
     * @param props DynamoPutItemProps
     */
    static putItem(props: DynamoPutItemProps): DynamoPutItem;
    /**
     * Method to get DynamoDeleteItem task
     *
     * @param props DynamoDeleteItemProps
     */
    static deleteItem(props: DynamoDeleteItemProps): DynamoDeleteItem;
    /**
     * Method to get DynamoUpdateItem task
     *
     * @param props DynamoUpdateItemProps
     */
    static updateItem(props: DynamoUpdateItemProps): DynamoUpdateItem;
}
