"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_b = props.containerOverrides) === null || _b === void 0 ? void 0 : _b.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    renderTask() {
        var _a;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_a = this.props.payload) === null || _a === void 0 ? void 0 : _a.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a, _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: (_b = containerOverrides.memory) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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