"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("../../aws-sns"); // Automatically re-written from '@aws-cdk/aws-sns'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Use a URL as a subscription target
 *
 * The message will be POSTed to the given URL.
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-http-https-endpoint-as-subscriber.html
 */
class UrlSubscription {
    constructor(url, props = {}) {
        this.url = url;
        this.props = props;
        this.unresolvedUrl = core_1.Token.isUnresolved(url);
        if (!this.unresolvedUrl && !url.startsWith('http://') && !url.startsWith('https://')) {
            throw new Error('URL must start with either http:// or https://');
        }
        if (this.unresolvedUrl && props.protocol === undefined) {
            throw new Error('Must provide protocol if url is unresolved');
        }
        if (this.unresolvedUrl) {
            this.protocol = props.protocol;
        }
        else {
            this.protocol = this.url.startsWith('https:') ? sns.SubscriptionProtocol.HTTPS : sns.SubscriptionProtocol.HTTP;
        }
    }
    /**
     * Returns a configuration for a URL to subscribe to an SNS topic
     */
    bind(_topic) {
        return {
            subscriberId: this.url,
            endpoint: this.url,
            protocol: this.protocol,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
}
exports.UrlSubscription = UrlSubscription;
//# sourceMappingURL=data:application/json;base64,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