"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const instance_1 = require("./instance");
const service_1 = require("./service");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
/**
 * Instance that is accessible using an IP address.
 *
 * @resource AWS::ServiceDiscovery::Instance
 */
class IpInstance extends instance_1.InstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const dnsRecordType = props.service.dnsRecordType;
        if (dnsRecordType === service_1.DnsRecordType.CNAME) {
            throw new Error('Service must support `A`, `AAAA` or `SRV` records to register this instance type.');
        }
        if (dnsRecordType === service_1.DnsRecordType.SRV) {
            if (!props.port) {
                throw new Error('A `port` must be specified for a service using a `SRV` record.');
            }
            if (!props.ipv4 && !props.ipv6) {
                throw new Error('At least `ipv4` or `ipv6` must be specified for a service using a `SRV` record.');
            }
        }
        if (!props.ipv4 && (dnsRecordType === service_1.DnsRecordType.A || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv4` must be specified for a service using a `A` record.');
        }
        if (!props.ipv6 &&
            (dnsRecordType === service_1.DnsRecordType.AAAA || dnsRecordType === service_1.DnsRecordType.A_AAAA)) {
            throw new Error('An `ipv6` must be specified for a service using a `AAAA` record.');
        }
        const port = props.port || 80;
        const resource = new servicediscovery_generated_1.CfnInstance(this, 'Resource', {
            instanceAttributes: {
                AWS_INSTANCE_IPV4: props.ipv4,
                AWS_INSTANCE_IPV6: props.ipv6,
                AWS_INSTANCE_PORT: port.toString(),
                ...props.customAttributes,
            },
            instanceId: props.instanceId || this.uniqueInstanceId(),
            serviceId: props.service.serviceId,
        });
        this.service = props.service;
        this.instanceId = resource.ref;
        this.ipv4 = props.ipv4 || '';
        this.ipv6 = props.ipv6 || '';
        this.port = port;
    }
}
exports.IpInstance = IpInstance;
//# sourceMappingURL=data:application/json;base64,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