"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A Lambda-based custom resource handler that provisions S3 bucket
 * notifications for a bucket.
 *
 * The resource property schema is:
 *
 * {
 *   BucketName: string, NotificationConfiguration: { see
 *   PutBucketNotificationConfiguration }
 * }
 *
 * For 'Delete' operations, we send an empty NotificationConfiguration as
 * required. We propagate errors and results as-is.
 *
 * Sadly, we can't use @aws-cdk/aws-lambda as it will introduce a dependency
 * cycle, so this uses raw `cdk.Resource`s.
 */
class NotificationsResourceHandler extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const role = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        // handler allows to put bucket notification on s3 buckets.
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:PutBucketNotification'],
            resources: ['*'],
        }));
        const resourceType = 'AWS::Lambda::Function';
        class InLineLambda extends cdk.CfnResource {
            constructor() {
                super(...arguments);
                this.tags = new cdk.TagManager(cdk.TagType.STANDARD, resourceType);
            }
            renderProperties(properties) {
                properties.Tags = cdk.listMapper(cdk.cfnTagToCloudFormation)(this.tags.renderTags());
                delete properties.tags;
                return properties;
            }
        }
        const resource = new InLineLambda(this, 'Resource', {
            type: resourceType,
            properties: {
                Description: 'AWS CloudFormation handler for "Custom::S3BucketNotifications" resources (@aws-cdk/aws-s3)',
                Code: { ZipFile: `exports.handler = ${handler.toString()};` },
                Handler: 'index.handler',
                Role: role.roleArn,
                Runtime: 'nodejs10.x',
                Timeout: 300,
            },
        });
        resource.node.addDependency(role);
        this.functionArn = resource.getAtt('Arn').toString();
    }
    /**
     * Defines a stack-singleton lambda function with the logic for a CloudFormation custom
     * resource that provisions bucket notification configuration for a bucket.
     *
     * @returns The ARN of the custom resource lambda function.
     */
    static singleton(context) {
        const root = cdk.Stack.of(context);
        // well-known logical id to ensure stack singletonity
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        let lambda = root.node.tryFindChild(logicalId);
        if (!lambda) {
            lambda = new NotificationsResourceHandler(root, logicalId);
        }
        return lambda.functionArn;
    }
}
exports.NotificationsResourceHandler = NotificationsResourceHandler;
// tslint:disable:no-console
/**
 * Lambda event handler for the custom resource. Bear in mind that we are going
 * to .toString() this function and inline it as Lambda code.
 *
 * The function will issue a putBucketNotificationConfiguration request for the
 * specified bucket.
 */
const handler = (event, context) => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports, import/no-extraneous-dependencies
    const s3 = new (require('aws-sdk').S3)();
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const https = require('https');
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const url = require('url');
    log(JSON.stringify(event, undefined, 2));
    const props = event.ResourceProperties;
    if (event.RequestType === 'Delete') {
        props.NotificationConfiguration = {}; // this is how you clean out notifications
    }
    const req = {
        Bucket: props.BucketName,
        NotificationConfiguration: props.NotificationConfiguration,
    };
    return s3.putBucketNotificationConfiguration(req, (err, data) => {
        log({ err, data });
        if (err) {
            return submitResponse('FAILED', err.message + `\nMore information in CloudWatch Log Stream: ${context.logStreamName}`);
        }
        else {
            return submitResponse('SUCCESS');
        }
    });
    function log(obj) {
        console.error(event.RequestId, event.StackId, event.LogicalResourceId, obj);
    }
    // tslint:disable-next-line:max-line-length
    // adapted from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-lambda-function-code.html#cfn-lambda-function-code-cfnresponsemodule
    // to allow sending an error messge as a reason.
    function submitResponse(responseStatus, reason) {
        const responseBody = JSON.stringify({
            Status: responseStatus,
            Reason: reason || 'See the details in CloudWatch Log Stream: ' + context.logStreamName,
            PhysicalResourceId: event.PhysicalResourceId || event.LogicalResourceId,
            StackId: event.StackId,
            RequestId: event.RequestId,
            LogicalResourceId: event.LogicalResourceId,
            NoEcho: false,
        });
        log({ responseBody });
        const parsedUrl = url.parse(event.ResponseURL);
        const options = {
            hostname: parsedUrl.hostname,
            port: 443,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': responseBody.length,
            },
        };
        const request = https.request(options, (r) => {
            log({ statusCode: r.statusCode, statusMessage: r.statusMessage });
            context.done();
        });
        request.on('error', (error) => {
            log({ sendError: error });
            context.done();
        });
        request.write(responseBody);
        request.end();
    }
};
//# sourceMappingURL=data:application/json;base64,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