"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Validates a zone name is valid by Route53 specifc naming rules,
 * and that there is no trailing dot in the name.
 *
 * @param zoneName the zone name to be validated.
 * @returns +zoneName+
 * @throws ValidationError if the name is not valid.
 */
function validateZoneName(zoneName) {
    if (zoneName.endsWith('.')) {
        throw new ValidationError('zone name must not end with a trailing dot');
    }
    if (zoneName.length > 255) {
        throw new ValidationError('zone name cannot be more than 255 bytes long');
    }
    if (zoneName.split('.').find(label => label.length > 63)) {
        throw new ValidationError('zone name labels cannot be more than 63 bytes long');
    }
    if (!zoneName.match(/^[a-z0-9!"#$%&'()*+,/:;<=>?@[\\\]^_`{|}~.-]+$/i)) {
        throw new ValidationError('zone names can only contain a-z, 0-9, -, ! " # $ % & \' ( ) * + , - / : ; < = > ? @ [ \ ] ^ _ ` { | } ~ .');
    }
}
exports.validateZoneName = validateZoneName;
class ValidationError extends Error {
    constructor(message) {
        super(message);
    }
}
/**
 * Route53 requires the record names are specified as fully qualified names, but this
 * forces lots of redundant work on the user (repeating the zone name over and over).
 * This function allows the user to be lazier and offers a nicer experience, by
 * qualifying relative names appropriately:
 *
 * @param providedName the user-specified name of the record.
 * @param zoneName   the fully-qualified name of the zone the record will be created in.
 *
 * @returns <ul>
 *        <li>If +providedName+ ends with a +.+, use it as-is</li>
 *        <li>If +providedName+ ends with or equals +zoneName+, append a trailing +.+</li>
 *        <li>Otherwise, append +.+, +zoneName+ and a trailing +.+</li>
 *      </ul>
 */
function determineFullyQualifiedDomainName(providedName, hostedZone) {
    if (providedName.endsWith('.')) {
        return providedName;
    }
    const hostedZoneName = hostedZone.zoneName.endsWith('.')
        ? hostedZone.zoneName.substring(0, hostedZone.zoneName.length - 1)
        : hostedZone.zoneName;
    const suffix = `.${hostedZoneName}`;
    if (providedName.endsWith(suffix) || providedName === hostedZoneName) {
        return `${providedName}.`;
    }
    return `${providedName}${suffix}.`;
}
exports.determineFullyQualifiedDomainName = determineFullyQualifiedDomainName;
function makeHostedZoneArn(construct, hostedZoneId) {
    return core_1.Stack.of(construct).formatArn({
        account: '',
        region: '',
        service: 'route53',
        resource: 'hostedzone',
        resourceName: hostedZoneId,
    });
}
exports.makeHostedZoneArn = makeHostedZoneArn;
//# sourceMappingURL=data:application/json;base64,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