"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const logs_generated_1 = require("./logs.generated");
/**
 * A new CloudWatch Logs Destination for use in cross-account scenarios
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * Policy object of this CrossAccountDestination object
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({ produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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