"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN. Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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