"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Builder
 */
class Builder {
    constructor(options) {
        this.options = options;
        // Original package.json
        const pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.originalPkg = fs.readFileSync(this.pkgPath);
        this.originalPkgJson = JSON.parse(this.originalPkg.toString());
    }
    /**
     * Build with parcel in a Docker container
     */
    build() {
        var _a, _b, _c, _d;
        try {
            this.updatePkg();
            const dockerBuildArgs = [
                'build',
                '--build-arg', `NODE_TAG=${this.options.nodeDockerTag}`,
                '-t', 'parcel-bundler',
                path.join(__dirname, '../parcel-bundler'),
            ];
            const build = child_process_1.spawnSync('docker', dockerBuildArgs);
            if (build.error) {
                throw build.error;
            }
            if (build.status !== 0) {
                throw new Error(`[Status ${build.status}] stdout: ${(_a = build.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = build.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
            }
            const containerProjectRoot = '/project';
            const containerOutDir = '/out';
            const containerCacheDir = '/cache';
            const containerEntryPath = path.join(containerProjectRoot, path.relative(this.options.projectRoot, path.resolve(this.options.entry)));
            const dockerRunArgs = [
                'run', '--rm',
                '-v', `${this.options.projectRoot}:${containerProjectRoot}`,
                '-v', `${path.resolve(this.options.outDir)}:${containerOutDir}`,
                ...(this.options.cacheDir ? ['-v', `${path.resolve(this.options.cacheDir)}:${containerCacheDir}`] : []),
                ...flatten(Object.entries(this.options.environment || {}).map(([k, v]) => ['--env', `${k}=${v}`])),
                '-w', path.dirname(containerEntryPath).replace(/\\/g, '/'),
                'parcel-bundler',
            ];
            const parcelArgs = [
                'parcel', 'build', containerEntryPath.replace(/\\/g, '/'),
                '--out-dir', containerOutDir,
                '--out-file', 'index.js',
                '--global', this.options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !this.options.minify && '--no-minify',
                !this.options.sourceMaps && '--no-source-maps',
                ...(this.options.cacheDir ? ['--cache-dir', containerCacheDir] : []),
            ].filter(Boolean);
            const parcel = child_process_1.spawnSync('docker', [...dockerRunArgs, ...parcelArgs]);
            if (parcel.error) {
                throw parcel.error;
            }
            if (parcel.status !== 0) {
                throw new Error(`[Status ${parcel.status}] stdout: ${(_c = parcel.stdout) === null || _c === void 0 ? void 0 : _c.toString().trim()}\n\n\nstderr: ${(_d = parcel.stderr) === null || _d === void 0 ? void 0 : _d.toString().trim()}`);
            }
        }
        catch (err) {
            throw new Error(`Failed to build file at ${this.options.entry}: ${err}`);
        }
        finally { // Always restore package.json to original
            this.restorePkg();
        }
    }
    /**
     * Updates the package.json to configure Parcel
     */
    updatePkg() {
        const updateData = {};
        // Update engines.node (Babel target)
        updateData.engines = { node: `>= ${this.options.nodeVersion}` };
        // Write new package.json
        if (Object.keys(updateData).length !== 0) {
            fs.writeFileSync(this.pkgPath, JSON.stringify({
                ...this.originalPkgJson,
                ...updateData,
            }, null, 2));
        }
    }
    restorePkg() {
        fs.writeFileSync(this.pkgPath, this.originalPkg);
    }
}
exports.Builder = Builder;
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
//# sourceMappingURL=data:application/json;base64,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