"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
            throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,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