"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    get keyId() {
        return this.aliasName;
    }
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias
     */
    static fromAliasAttributes(scope, id, attrs) {
        // tslint:disable-next-line: class-name
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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