"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        this.resource.node.addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidW5rbm93bi1wcmluY2lwYWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ1bmtub3duLXByaW5jaXBhbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHFDQUFtRSxDQUFDLGdEQUFnRDtBQVlwSDs7Ozs7Ozs7OztHQVVHO0FBQ0gsTUFBYSxnQkFBZ0I7SUFJekIsWUFBWSxLQUE0QjtRQUh4QixxQkFBZ0IsR0FBVyxnQkFBZ0IsQ0FBQztRQUl4RCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUM7SUFDL0IsQ0FBQztJQUNELElBQVcsY0FBYztRQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxJQUFJLG9DQUFvQyxDQUFDLENBQUM7SUFDbEgsQ0FBQztJQUNNLG9CQUFvQixDQUFDLFNBQTBCO1FBQ2xELE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQywwQ0FBMEMsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNoRixvRUFBb0U7UUFDcEUsT0FBTyxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsSUFBSSx5QkFBa0IsRUFBRSxFQUFFLENBQUM7SUFDaEYsQ0FBQztJQUNNLFdBQVcsQ0FBQyxTQUEwQjtRQUN6QyxPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsQ0FBQyxjQUFjLENBQUM7SUFDL0QsQ0FBQztDQUNKO0FBckJELDRDQXFCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbmNyZXRlRGVwZW5kYWJsZSwgSUNvbnN0cnVjdCwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4vcG9saWN5LXN0YXRlbWVudCc7XG5pbXBvcnQgeyBBZGRUb1ByaW5jaXBhbFBvbGljeVJlc3VsdCwgSVByaW5jaXBhbCwgUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQgfSBmcm9tICcuL3ByaW5jaXBhbHMnO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhbiBVbmtub3duUHJpbmNpcGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVW5rbm93blByaW5jaXBhbFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2UgdGhlIHJvbGUgcHJveHkgaXMgZm9yXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2U6IElDb25zdHJ1Y3Q7XG59XG4vKipcbiAqIEEgcHJpbmNpcGFsIGZvciB1c2UgaW4gcmVzb3VyY2VzIHRoYXQgbmVlZCB0byBoYXZlIGEgcm9sZSBidXQgaXQncyB1bmtub3duXG4gKlxuICogU29tZSByZXNvdXJjZXMgaGF2ZSByb2xlcyBhc3NvY2lhdGVkIHdpdGggdGhlbSB3aGljaCB0aGV5IGFzc3VtZSwgc3VjaCBhc1xuICogTGFtYmRhIEZ1bmN0aW9ucywgQ29kZUJ1aWxkIHByb2plY3RzLCBTdGVwRnVuY3Rpb25zIG1hY2hpbmVzLCBldGMuXG4gKlxuICogV2hlbiB0aG9zZSByZXNvdXJjZXMgYXJlIGltcG9ydGVkLCB0aGVpciBhY3R1YWwgcm9sZXMgYXJlIG5vdCBhbHdheXNcbiAqIGltcG9ydGVkIHdpdGggdGhlbS4gV2hlbiB0aGF0IGhhcHBlbnMsIHdlIHVzZSBhbiBpbnN0YW5jZSBvZiB0aGlzIGNsYXNzXG4gKiBpbnN0ZWFkLCB3aGljaCB3aWxsIGFkZCB1c2VyIHdhcm5pbmdzIHdoZW4gc3RhdGVtZW50cyBhcmUgYXR0ZW1wdGVkIHRvIGJlXG4gKiBhZGRlZCB0byBpdC5cbiAqL1xuZXhwb3J0IGNsYXNzIFVua25vd25QcmluY2lwYWwgaW1wbGVtZW50cyBJUHJpbmNpcGFsIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgYXNzdW1lUm9sZUFjdGlvbjogc3RyaW5nID0gJ3N0czpBc3N1bWVSb2xlJztcbiAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IElQcmluY2lwYWw7XG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogSUNvbnN0cnVjdDtcbiAgICBjb25zdHJ1Y3Rvcihwcm9wczogVW5rbm93blByaW5jaXBhbFByb3BzKSB7XG4gICAgICAgIHRoaXMucmVzb3VyY2UgPSBwcm9wcy5yZXNvdXJjZTtcbiAgICAgICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXM7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgcG9saWN5RnJhZ21lbnQoKTogUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBnZXQgcG9saWN5IGZyYWdtZW50IG9mICR7dGhpcy5yZXNvdXJjZS5ub2RlLnBhdGh9LCByZXNvdXJjZSBpbXBvcnRlZCB3aXRob3V0IGEgcm9sZWApO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBBZGRUb1ByaW5jaXBhbFBvbGljeVJlc3VsdCB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcy5yZXNvdXJjZSk7XG4gICAgICAgIGNvbnN0IHJlcHIgPSBKU09OLnN0cmluZ2lmeShzdGFjay5yZXNvbHZlKHN0YXRlbWVudCkpO1xuICAgICAgICB0aGlzLnJlc291cmNlLm5vZGUuYWRkV2FybmluZyhgQWRkIHN0YXRlbWVudCB0byB0aGlzIHJlc291cmNlJ3Mgcm9sZTogJHtyZXByfWApO1xuICAgICAgICAvLyBQcmV0ZW5kIHdlIGRpZCB0aGUgd29yay4gVGhlIGh1bWFuIHdpbGwgZG8gaXQgZm9yIHVzLCBldmVudHVhbGx5LlxuICAgICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogdHJ1ZSwgcG9saWN5RGVwZW5kYWJsZTogbmV3IENvbmNyZXRlRGVwZW5kYWJsZSgpIH07XG4gICAgfVxuICAgIHB1YmxpYyBhZGRUb1BvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IGJvb2xlYW4ge1xuICAgICAgICByZXR1cm4gdGhpcy5hZGRUb1ByaW5jaXBhbFBvbGljeShzdGF0ZW1lbnQpLnN0YXRlbWVudEFkZGVkO1xuICAgIH1cbn1cbiJdfQ==