"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        var _a;
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? (_a = resourceResult.policyDependable) !== null && _a !== void 0 ? _a : options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
//# sourceMappingURL=data:application/json;base64,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