"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (props.s3Prefix !== undefined && props.s3Prefix !== null) ? props.s3Prefix : 'data/';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed === undefined ? false : props.compressed;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories === undefined ? false : props.storedAsSubDirectories,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).parseArn(tableArn).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.s3Prefix);
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
}
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error('column names and partition keys must be unique, but \'p1\' is duplicated');
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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