"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/data-types.html
 */
class Schema {
    /**
     * Creates a decimal type.
     *
     * TODO: Bounds
     *
     * @param precision the total number of digits
     * @param scale the number of digits in fractional part, the default is 0
     */
    static decimal(precision, scale) {
        return {
            isPrimitive: true,
            inputString: scale !== undefined ? `decimal(${precision},${scale})` : `decimal(${precision})`,
        };
    }
    /**
     * Fixed length character data, with a specified length between 1 and 255.
     *
     * @param length length between 1 and 255
     */
    static char(length) {
        if (length <= 0 || length > 255) {
            throw new Error(`char length must be (inclusively) between 1 and 255, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`char length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `char(${length})`,
        };
    }
    /**
     * Variable length character data, with a specified length between 1 and 65535.
     *
     * @param length length between 1 and 65535.
     */
    static varchar(length) {
        if (length <= 0 || length > 65535) {
            throw new Error(`varchar length must be (inclusively) between 1 and 65535, but was ${length}`);
        }
        if (length % 1 !== 0) {
            throw new Error(`varchar length must be a positive integer, was ${length}`);
        }
        return {
            isPrimitive: true,
            inputString: `varchar(${length})`,
        };
    }
    /**
     * Creates an array of some other type.
     *
     * @param itemType type contained by the array.
     */
    static array(itemType) {
        return {
            isPrimitive: false,
            inputString: `array<${itemType.inputString}>`,
        };
    }
    /**
     * Creates a map of some primitive key type to some value type.
     *
     * @param keyType type of key, must be a primitive.
     * @param valueType type fo the value indexed by the key.
     */
    static map(keyType, valueType) {
        if (!keyType.isPrimitive) {
            throw new Error(`the key type of a 'map' must be a primitive, but was ${keyType.inputString}`);
        }
        return {
            isPrimitive: false,
            inputString: `map<${keyType.inputString},${valueType.inputString}>`,
        };
    }
    /**
     * Creates a nested structure containing individually named and typed columns.
     *
     * @param columns the columns of the structure.
     */
    static struct(columns) {
        return {
            isPrimitive: false,
            inputString: `struct<${columns.map(column => {
                if (column.comment === undefined) {
                    return `${column.name}:${column.type.inputString}`;
                }
                else {
                    return `${column.name}:${column.type.inputString} COMMENT '${column.comment}'`;
                }
            }).join(',')}>`,
        };
    }
}
exports.Schema = Schema;
Schema.BOOLEAN = {
    isPrimitive: true,
    inputString: 'boolean',
};
Schema.BINARY = {
    isPrimitive: true,
    inputString: 'binary',
};
/**
 * A 64-bit signed INTEGER in two’s complement format, with a minimum value of -2^63 and a maximum value of 2^63-1.
 */
Schema.BIG_INT = {
    isPrimitive: true,
    inputString: 'bigint',
};
Schema.DOUBLE = {
    isPrimitive: true,
    inputString: 'double',
};
Schema.FLOAT = {
    isPrimitive: true,
    inputString: 'float',
};
/**
 * A 32-bit signed INTEGER in two’s complement format, with a minimum value of -2^31 and a maximum value of 2^31-1.
 */
Schema.INTEGER = {
    isPrimitive: true,
    inputString: 'int',
};
/**
 * A 16-bit signed INTEGER in two’s complement format, with a minimum value of -2^15 and a maximum value of 2^15-1.
 */
Schema.SMALL_INT = {
    isPrimitive: true,
    inputString: 'smallint',
};
/**
 * A 8-bit signed INTEGER in two’s complement format, with a minimum value of -2^7 and a maximum value of 2^7-1
 */
Schema.TINY_INT = {
    isPrimitive: true,
    inputString: 'tinyint',
};
/**
 * Date type.
 */
Schema.DATE = {
    isPrimitive: true,
    inputString: 'date',
};
/**
 * Timestamp type (date and time).
 */
Schema.TIMESTAMP = {
    isPrimitive: true,
    inputString: 'timestamp',
};
/**
 * Arbitrary-length string type.
 */
Schema.STRING = {
    isPrimitive: true,
    inputString: 'string',
};
//# sourceMappingURL=data:application/json;base64,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