"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * InputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
 */
InputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerInputFormat');
/**
 * InputFormat for Cloudtrail Logs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
InputFormat.CLOUDTRAIL = new InputFormat('com.amazon.emr.cloudtrail.CloudTrailInputFormat');
/**
 * InputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
 */
InputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcInputFormat');
/**
 * InputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
 */
InputFormat.PARQUET = new InputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat');
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 * JSON & CSV files are examples of this InputFormat
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * OutputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
 */
OutputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat');
/**
 * OutputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
 */
OutputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat');
/**
 * OutputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
 */
OutputFormat.PARQUET = new OutputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
 */
SerializationLibrary.AVRO = new SerializationLibrary('org.apache.hadoop.hive.serde2.avro.AvroSerDe');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
SerializationLibrary.CLOUDTRAIL = new SerializationLibrary('com.amazon.emr.hive.serde.CloudTrailSerde');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
SerializationLibrary.GROK = new SerializationLibrary('com.amazonaws.glue.serde.GrokSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
 */
SerializationLibrary.LAZY_SIMPLE = new SerializationLibrary('org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
 */
SerializationLibrary.OPEN_CSV = new SerializationLibrary('org.apache.hadoop.hive.serde2.OpenCSVSerde');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
 */
SerializationLibrary.ORC = new SerializationLibrary('org.apache.hadoop.hive.ql.io.orc.OrcSerde');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
 */
SerializationLibrary.PARQUET = new SerializationLibrary('org.apache.hadoop.hive.ql.io.parquet.serde.ParquetHiveSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
 */
SerializationLibrary.REGEXP = new SerializationLibrary('org.apache.hadoop.hive.serde2.RegexSerDe');
/**
 * Defines the input/output formats and ser/de for a single DataFormat.
 */
class DataFormat {
    constructor(props) {
        this.inputFormat = props.inputFormat;
        this.outputFormat = props.outputFormat;
        this.serializationLibrary = props.serializationLibrary;
    }
}
exports.DataFormat = DataFormat;
/**
 * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
 */
DataFormat.APACHE_LOGS = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.REGEXP,
});
/**
 * DataFormat for Apache Avro
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
 */
DataFormat.AVRO = new DataFormat({
    inputFormat: InputFormat.AVRO,
    outputFormat: OutputFormat.AVRO,
    serializationLibrary: SerializationLibrary.AVRO,
});
/**
 * DataFormat for CloudTrail logs stored on S3
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
DataFormat.CLOUDTRAIL_LOGS = new DataFormat({
    inputFormat: InputFormat.CLOUDTRAIL,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.CLOUDTRAIL,
});
/**
 * DataFormat for CSV Files
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
 */
DataFormat.CSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPEN_CSV,
});
/**
 * Stored as plain text files in JSON format.
 * Uses OpenX Json SerDe for serialization and deseralization.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
 */
DataFormat.JSON = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPENX_JSON,
});
/**
 * DataFormat for Logstash Logs, using the GROK SerDe
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
DataFormat.LOGSTASH = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.GROK,
});
/**
 * DataFormat for Apache ORC (Optimized Row Columnar)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
 */
DataFormat.ORC = new DataFormat({
    inputFormat: InputFormat.ORC,
    outputFormat: OutputFormat.ORC,
    serializationLibrary: SerializationLibrary.ORC,
});
/**
 * DataFormat for Apache Parquet
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
 */
DataFormat.PARQUET = new DataFormat({
    inputFormat: InputFormat.PARQUET,
    outputFormat: OutputFormat.PARQUET,
    serializationLibrary: SerializationLibrary.PARQUET,
});
/**
 * DataFormat for TSV (Tab-Separated Values)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
 */
DataFormat.TSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.LAZY_SIMPLE,
});
//# sourceMappingURL=data:application/json;base64,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