"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines a CloudWatch Event Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.anyValue({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.anyValue({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing CloudWatch Event Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = targetProps.role ? targetProps.role.roleArn : undefined;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/CloudWatchEvents-CrossAccountEventDelivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (CloudWatch verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = targetProps.targetResource.node.root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${this.node.uniqueId}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: inputProps && inputProps.inputTemplate !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
//# sourceMappingURL=data:application/json;base64,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