"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
/**
 * Use an SQS Queue as a target for AWS CloudWatch event rules.
 *
 * @example
 *
 *    // publish to an SQS queue every time code is committed
 *    // to a CodeCommit repository
 *    repository.onCommit(new targets.SqsQueue(queue));
 *
 */
class SqsQueue {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        if (props.messageGroupId !== undefined && !queue.fifo) {
            throw new Error('messageGroupId cannot be specified for non-FIFO queues');
        }
    }
    /**
     * Returns a RuleTarget that can be used to trigger this SQS queue as a
     * result from a CloudWatch event.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/resource-based-policies-cwe.html#sqs-permissions
     */
    bind(rule, _id) {
        // deduplicated automatically
        this.queue.grantSendMessages(new iam.ServicePrincipal('events.amazonaws.com', {
            conditions: {
                ArnEquals: { 'aws:SourceArn': rule.ruleArn },
            },
        }));
        return {
            id: '',
            arn: this.queue.queueArn,
            input: this.props.message,
            targetResource: this.queue,
            sqsParameters: this.props.messageGroupId ? { messageGroupId: this.props.messageGroupId } : undefined,
        };
    }
}
exports.SqsQueue = SqsQueue;
//# sourceMappingURL=data:application/json;base64,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