"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 */
class NetworkListenerAction {
    /**
     * Create an instance of NetworkListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Forward to one or more Target Groups
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `NetworkListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.NetworkListenerAction = NetworkListenerAction;
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends NetworkListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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