"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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