"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => ({
                rolearn: m.role.roleArn,
                username: m.mapping.username,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => ({
                userarn: m.user.userArn,
                username: m.mapping.username,
                groups: m.mapping.groups,
            }))),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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