"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const efs_generated_1 = require("./efs.generated");
// tslint:disable:max-line-length
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
// tslint:enable
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_7_DAYS"] = 0] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_14_DAYS"] = 1] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_30_DAYS"] = 2] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_60_DAYS"] = 3] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy[LifecyclePolicy["AFTER_90_DAYS"] = 4] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-efs-filesystem-performancemode
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * This is the general purpose performance mode for most file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * This performance mode can scale to higher levels of aggregate throughput and operations per second with a
     * tradeoff of slightly higher latencies.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-throughputmode
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     *  This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends core_1.Resource {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: props.encrypted,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lifecyclePolicies: (props.lifecyclePolicy ? Array.of({
                transitionToIa: LifecyclePolicy[props.lifecyclePolicy],
            }) : undefined),
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: (_a = props.provisionedThroughputPerSecond) === null || _a === void 0 ? void 0 : _a.toMebibytes(),
        });
        this.fileSystemId = filesystem.ref;
        core_1.Tag.add(this, 'Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets);
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        subnets.subnetIds.forEach((subnetId) => {
            new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
        });
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.fileSystemId = attrs.fileSystemId;
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.FileSystem = FileSystem;
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
//# sourceMappingURL=data:application/json;base64,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