"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const proxy_configuration_1 = require("./proxy-configuration");
/**
 * The class for App Mesh proxy configurations.
 *
 * For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent and at least version
 * 1.26.0-1 of the ecs-init package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized
 * AMI version 20190301 or later, then they contain the required versions of the container agent and ecs-init.
 * For more information, see [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 *
 * For tasks using the Fargate launch type, the task or service requires platform version 1.3.0 or later.
 */
class AppMeshProxyConfiguration extends proxy_configuration_1.ProxyConfiguration {
    /**
     * Constructs a new instance of the AppMeshProxyConfiguration class.
     */
    constructor(props) {
        super();
        this.props = props;
        if (props.properties) {
            if (!props.properties.ignoredUID && !props.properties.ignoredGID) {
                throw new Error('At least one of ignoredUID or ignoredGID should be specified.');
            }
        }
    }
    /**
     * Called when the proxy configuration is configured on a task definition.
     */
    bind(_scope, _taskDefinition) {
        const configProps = this.props.properties;
        const configType = 'APPMESH';
        return {
            containerName: this.props.containerName,
            proxyConfigurationProperties: renderProperties(configProps),
            type: configType,
        };
    }
}
exports.AppMeshProxyConfiguration = AppMeshProxyConfiguration;
function renderProperties(props) {
    const ret = new Array();
    for (const [k, v] of Object.entries(props)) {
        const key = String(k);
        const value = String(v);
        if (value !== 'undefined' && value !== '') {
            const capitalizedKey = key.charAt(0).toUpperCase() + key.slice(1);
            ret.push({ ['name']: capitalizedKey, ['value']: value });
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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