"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2_service_1 = require("./ec2/ec2-service");
/**
 * Instance resource used for bin packing
 */
var BinPackResource;
(function (BinPackResource) {
    /**
     * Fill up hosts' CPU allocations first
     */
    BinPackResource["CPU"] = "cpu";
    /**
     * Fill up hosts' memory allocations first
     */
    BinPackResource["MEMORY"] = "memory";
})(BinPackResource = exports.BinPackResource || (exports.BinPackResource = {}));
/**
 * The placement strategies to use for tasks in the service. For more information, see
 * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
 *
 * Tasks will preferentially be placed on instances that match these rules.
 */
class PlacementStrategy {
    /**
     * Constructs a new instance of the PlacementStrategy class.
     */
    constructor(json) {
        this.json = json;
    }
    /**
     * Places tasks evenly across all container instances in the cluster.
     */
    static spreadAcrossInstances() {
        return new PlacementStrategy([{ type: 'spread', field: ec2_service_1.BuiltInAttributes.INSTANCE_ID }]);
    }
    /**
     * Places tasks evenly based on the specified value.
     *
     * You can use one of the built-in attributes found on `BuiltInAttributes`
     * or supply your own custom instance attributes. If more than one attribute
     * is supplied, spreading is done in order.
     *
     * @default attributes instanceId
     */
    static spreadAcross(...fields) {
        if (fields.length === 0) {
            throw new Error('spreadAcross: give at least one field to spread by');
        }
        return new PlacementStrategy(fields.map(field => ({ type: 'spread', field })));
    }
    /**
     * Places tasks on container instances with the least available amount of CPU capacity.
     *
     * This minimizes the number of instances in use.
     */
    static packedByCpu() {
        return PlacementStrategy.packedBy(BinPackResource.CPU);
    }
    /**
     * Places tasks on container instances with the least available amount of memory capacity.
     *
     * This minimizes the number of instances in use.
     */
    static packedByMemory() {
        return PlacementStrategy.packedBy(BinPackResource.MEMORY);
    }
    /**
     * Places tasks on the container instances with the least available capacity of the specified resource.
     */
    static packedBy(resource) {
        return new PlacementStrategy([{ type: 'binpack', field: resource }]);
    }
    /**
     * Places tasks randomly.
     */
    static randomly() {
        return new PlacementStrategy([{ type: 'random' }]);
    }
    /**
     * Return the placement JSON
     */
    toJson() {
        return this.json;
    }
}
exports.PlacementStrategy = PlacementStrategy;
/**
 * The placement constraints to use for tasks in the service. For more information, see
 * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
 *
 * Tasks will only be placed on instances that match these rules.
 */
class PlacementConstraint {
    /**
     * Constructs a new instance of the PlacementConstraint class.
     */
    constructor(json) {
        this.json = json;
    }
    /**
     * Use distinctInstance to ensure that each task in a particular group is running on a different container instance.
     */
    static distinctInstances() {
        return new PlacementConstraint([{ type: 'distinctInstance' }]);
    }
    /**
     * Use memberOf to restrict the selection to a group of valid candidates specified by a query expression.
     *
     * Multiple expressions can be specified. For more information, see
     * [Cluster Query Language](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html).
     *
     * You can specify multiple expressions in one call. The tasks will only be placed on instances matching all expressions.
     *
     * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html
     */
    static memberOf(...expressions) {
        return new PlacementConstraint(expressions.map(expression => ({ type: 'memberOf', expression })));
    }
    /**
     * Return the placement JSON
     */
    toJson() {
        return this.json;
    }
}
exports.PlacementConstraint = PlacementConstraint;
//# sourceMappingURL=data:application/json;base64,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