"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_image_1 = require("../container-image");
/**
 * Regex pattern to check if it is an ECR image URL.
 *
 * @experimental
 */
const ECR_IMAGE_REGEX = /(^[a-zA-Z0-9][a-zA-Z0-9-_]*).dkr.ecr.([a-zA-Z0-9][a-zA-Z0-9-_]*).amazonaws.com(.cn)?\/.*/;
/**
 * An image hosted in a public or private repository. For images hosted in Amazon ECR, see
 * [EcrImage](https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html).
 */
class RepositoryImage extends container_image_1.ContainerImage {
    /**
     * Constructs a new instance of the RepositoryImage class.
     */
    constructor(imageName, props = {}) {
        super();
        this.imageName = imageName;
        this.props = props;
    }
    bind(scope, containerDefinition) {
        // name could be a Token - in that case, skip validation altogether
        if (!core_1.Token.isUnresolved(this.imageName) && ECR_IMAGE_REGEX.test(this.imageName)) {
            scope.node.addWarning("Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
        }
        if (this.props.credentials) {
            this.props.credentials.grantRead(containerDefinition.taskDefinition.obtainExecutionRole());
        }
        return {
            imageName: this.imageName,
            repositoryCredentials: this.props.credentials && {
                credentialsParameter: this.props.credentials.secretArn,
            },
        };
    }
}
exports.RepositoryImage = RepositoryImage;
//# sourceMappingURL=data:application/json;base64,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