"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
const task_definition_1 = require("../base/task-definition");
/**
 * This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @resource AWS::ECS::Service
 */
class Ec2Service extends base_service_1.BaseService {
    /**
     * Constructs a new instance of the Ec2Service class.
     */
    constructor(scope, id, props) {
        var _a;
        if (props.daemon && props.desiredCount !== undefined) {
            throw new Error('Daemon mode launches one task on every instance. Don\'t supply desiredCount.');
        }
        if (props.daemon && props.maxHealthyPercent !== undefined && props.maxHealthyPercent !== 100) {
            throw new Error('Maximum percent must be 100 for daemon mode.');
        }
        if (props.minHealthyPercent !== undefined && props.maxHealthyPercent !== undefined && props.minHealthyPercent >= props.maxHealthyPercent) {
            throw new Error('Minimum healthy percent must be less than maximum healthy percent.');
        }
        if (!props.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with EC2');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            // If daemon, desiredCount must be undefined and that's what we want. Otherwise, default to 1.
            desiredCount: props.daemon || props.desiredCount !== undefined ? props.desiredCount : 1,
            maxHealthyPercent: props.daemon && props.maxHealthyPercent === undefined ? 100 : props.maxHealthyPercent,
            minHealthyPercent: props.daemon && props.minHealthyPercent === undefined ? 0 : props.minHealthyPercent,
            launchType: base_service_1.LaunchType.EC2,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            placementConstraints: core_1.Lazy.anyValue({ produce: () => this.constraints }, { omitEmptyArray: true }),
            placementStrategies: core_1.Lazy.anyValue({ produce: () => this.strategies }, { omitEmptyArray: true }),
            schedulingStrategy: props.daemon ? 'DAEMON' : 'REPLICA',
        }, props.taskDefinition);
        this.constraints = [];
        this.strategies = [];
        this.daemon = props.daemon || false;
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        if (props.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroups from ASG.
            // We have to be smart here -- by default future Security Group rules would be created
            // in the Cluster stack. However, if the Cluster is in a different stack than us,
            // that will lead to a cyclic reference (we point to that stack for the cluster name,
            // but that stack will point to the ALB probably created right next to us).
            //
            // In that case, reference the same security groups but make sure new rules are
            // created in the current scope (i.e., this stack)
            validateNoNetworkingProps(props);
            this.connections.addSecurityGroup(...securityGroupsInThisStack(this, props.cluster.connections.securityGroups));
        }
        this.addPlacementConstraints(...props.placementConstraints || []);
        this.addPlacementStrategies(...props.placementStrategies || []);
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
    /**
     * Imports from the specified service ARN.
     */
    static fromEc2ServiceArn(scope, id, ec2ServiceArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = ec2ServiceArn;
                this.serviceName = core_1.Stack.of(scope).parseArn(ec2ServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports from the specified service attrributes.
     */
    static fromEc2ServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     */
    addPlacementStrategies(...strategies) {
        if (strategies.length > 0 && this.daemon) {
            throw new Error("Can't configure placement strategies when daemon=true");
        }
        for (const strategy of strategies) {
            this.strategies.push(...strategy.toJson());
        }
    }
    /**
     * Adds one or more placement strategies to use for tasks in the service. For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     */
    addPlacementConstraints(...constraints) {
        for (const constraint of constraints) {
            this.constraints.push(...constraint.toJson());
        }
    }
    /**
     * Validates this Ec2Service.
     */
    validate() {
        const ret = super.validate();
        if (!this.cluster.hasEc2Capacity) {
            ret.push('Cluster for this service needs Ec2 capacity. Call addXxxCapacity() on the cluster.');
        }
        return ret;
    }
}
exports.Ec2Service = Ec2Service;
/**
 * Validate combinations of networking arguments.
 */
function validateNoNetworkingProps(props) {
    if (props.vpcSubnets !== undefined
        || props.securityGroup !== undefined
        || props.securityGroups !== undefined
        || props.assignPublicIp) {
        throw new Error('vpcSubnets, securityGroup(s) and assignPublicIp can only be used in AwsVpc networking mode');
    }
}
/**
 * Force security group rules to be created in this stack.
 *
 * For every security group, if the scope and the group are in different stacks, return
 * a fake "imported" security group instead. This will behave as the original security group,
 * but new Ingress and Egress rule resources will be added in the current stack instead of the
 * other one.
 */
function securityGroupsInThisStack(scope, groups) {
    const thisStack = core_1.Stack.of(scope);
    let i = 1;
    return groups.map(group => {
        if (thisStack === core_1.Stack.of(group)) {
            return group;
        } // Simple case, just return the original one
        return ec2.SecurityGroup.fromSecurityGroupId(scope, `SecurityGroup${i++}`, group.securityGroupId, {
            allowAllOutbound: group.allowAllOutbound,
            mutable: true,
        });
    });
}
/**
 * The built-in container instance attributes
 */
class BuiltInAttributes {
}
exports.BuiltInAttributes = BuiltInAttributes;
/**
 * The id of the instance.
 */
BuiltInAttributes.INSTANCE_ID = 'instanceId';
/**
 * The AvailabilityZone where the instance is running in.
 */
BuiltInAttributes.AVAILABILITY_ZONE = 'attribute:ecs.availability-zone';
/**
 * The AMI id the instance is using.
 */
BuiltInAttributes.AMI_ID = 'attribute:ecs.ami-id';
/**
 * The EC2 instance type.
 */
BuiltInAttributes.INSTANCE_TYPE = 'attribute:ecs.instance-type';
/**
 * The operating system of the instance.
 *
 * Either 'linux' or 'windows'.
 */
BuiltInAttributes.OS_TYPE = 'attribute:ecs.os-type';
//# sourceMappingURL=data:application/json;base64,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