"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const autoscaling = require("../../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const hooks = require("../../../aws-autoscaling-hooktargets"); // Automatically re-written from '@aws-cdk/aws-autoscaling-hooktargets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
/**
 * A hook to drain instances from ECS traffic before they're terminated
 */
class InstanceDrainHook extends cdk.Construct {
    /**
     * Constructs a new instance of the InstanceDrainHook class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const drainTime = props.drainTime || cdk.Duration.minutes(5);
        // Invoke Lambda via SNS Topic
        const fn = new lambda.Function(this, 'Function', {
            code: lambda.Code.fromInline(fs.readFileSync(path.join(__dirname, 'lambda-source', 'index.py'), { encoding: 'utf-8' })),
            handler: 'index.lambda_handler',
            runtime: lambda.Runtime.PYTHON_3_6,
            // Timeout: some extra margin for additional API calls made by the Lambda,
            // up to a maximum of 15 minutes.
            timeout: cdk.Duration.seconds(Math.min(drainTime.toSeconds() + 10, 900)),
            environment: {
                CLUSTER: props.cluster.clusterName,
            },
        });
        // Hook everything up: ASG -> Topic, Topic -> Lambda
        props.autoScalingGroup.addLifecycleHook('DrainHook', {
            lifecycleTransition: autoscaling.LifecycleTransition.INSTANCE_TERMINATING,
            defaultResult: autoscaling.DefaultResult.CONTINUE,
            notificationTarget: new hooks.FunctionHook(fn),
            heartbeatTimeout: drainTime,
        });
        // Describe actions cannot be restricted and restrict the CompleteLifecycleAction to the ASG arn
        // https://docs.aws.amazon.com/autoscaling/ec2/userguide/control-access-using-iam.html
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeInstances',
                'ec2:DescribeInstanceAttribute',
                'ec2:DescribeInstanceStatus',
                'ec2:DescribeHosts',
            ],
            resources: ['*'],
        }));
        // Restrict to the ASG
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['autoscaling:CompleteLifecycleAction'],
            resources: [props.autoScalingGroup.autoScalingGroupArn],
        }));
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: ['ecs:DescribeContainerInstances', 'ecs:DescribeTasks'],
            resources: ['*'],
        }));
        // Restrict to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:ListContainerInstances',
                'ecs:SubmitContainerStateChange',
                'ecs:SubmitTaskStateChange',
            ],
            resources: [props.cluster.clusterArn],
        }));
        // Restrict the container-instance operations to the ECS Cluster
        fn.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:UpdateContainerInstancesState',
                'ecs:ListTasks',
            ],
            conditions: {
                ArnEquals: { 'ecs:cluster': props.cluster.clusterArn },
            },
            resources: ['*'],
        }));
    }
}
exports.InstanceDrainHook = InstanceDrainHook;
//# sourceMappingURL=data:application/json;base64,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