"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_generated_1 = require("./ecs.generated");
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        const containerInsights = props.containerInsights !== undefined ? props.containerInsights : false;
        const clusterSettings = containerInsights ? [{ name: 'containerInsights', value: 'enabled' }] : undefined;
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImage || new EcsOptimizedAmi(),
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(autoScalingGroup, options);
        return autoScalingGroup;
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        // Tie instances to cluster
        autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
        if (!options.canContainersAccessInstanceRole) {
            // Deny containers access to instance metadata service
            // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
            autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
            autoScalingGroup.addUserData('sudo service iptables save');
            // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
            autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
        }
        if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
            autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:CreateCluster',
                'ecs:DeregisterContainerInstance',
                'ecs:DiscoverPollEndpoint',
                'ecs:Poll',
                'ecs:RegisterContainerInstance',
                'ecs:StartTelemetrySession',
                'ecs:Submit*',
                'ecr:GetAuthorizationToken',
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
            });
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        return this.metric('CPUReservation', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        return this.metric('MemoryReservation', props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
/**
 * ECS-optimized Windows version list
 */
var WindowsOptimizedVersion;
(function (WindowsOptimizedVersion) {
    WindowsOptimizedVersion["SERVER_2019"] = "2019";
    WindowsOptimizedVersion["SERVER_2016"] = "2016";
})(WindowsOptimizedVersion = exports.WindowsOptimizedVersion || (exports.WindowsOptimizedVersion = {}));
/*
 * TODO:v2.0.0 remove EcsOptimizedAmi
 */
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
class EcsOptimizedAmi {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = (props && props.hardwareType) || AmiHardwareType.STANDARD;
        if (props && props.generation) { // generation defined in the props object
            if (props.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX && this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Amazon Linux does not support special hardware type. Use Amazon Linux 2 instead');
            }
            else if (props.windowsVersion) {
                throw new Error('"windowsVersion" and Linux image "generation" cannot be both set');
            }
            else {
                this.generation = props.generation;
            }
        }
        else if (props && props.windowsVersion) {
            if (this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Windows Server does not support special hardware type');
            }
            else {
                this.windowsVersion = props.windowsVersion;
            }
        }
        else { // generation not defined in props object
            // always default to Amazon Linux v2 regardless of HW
            this.generation = ec2.AmazonLinuxGeneration.AMAZON_LINUX_2;
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedAmi = EcsOptimizedAmi;
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 */
class EcsOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = props && props.hardwareType;
        if (props.windowsVersion) {
            this.windowsVersion = props.windowsVersion;
        }
        else if (props.generation) {
            this.generation = props.generation;
        }
        else {
            throw new Error('This error should never be thrown');
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM
     *
     * @param hardwareType ECS-optimized AMI variant to use
     */
    static amazonLinux2(hardwareType = AmiHardwareType.STANDARD) {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2, hardwareType });
    }
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM
     */
    static amazonLinux() {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX });
    }
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM
     *
     * @param windowsVersion Windows Version to use
     */
    static windows(windowsVersion) {
        return new EcsOptimizedImage({ windowsVersion });
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedImage = EcsOptimizedImage;
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this.clusterArn = props.clusterArn !== undefined ? props.clusterArn : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
}
/**
 * The ECS-optimized AMI variant to use. For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 */
var AmiHardwareType;
(function (AmiHardwareType) {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     */
    AmiHardwareType["STANDARD"] = "Standard";
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     */
    AmiHardwareType["GPU"] = "GPU";
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     */
    AmiHardwareType["ARM"] = "ARM64";
})(AmiHardwareType = exports.AmiHardwareType || (exports.AmiHardwareType = {}));
//# sourceMappingURL=data:application/json;base64,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