"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const network_multiple_target_groups_service_base_1 = require("../base/network-multiple-target-groups-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkMultipleTargetGroupsFargateService extends network_multiple_target_groups_service_base_1.NetworkMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: this.logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createFargateService(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createFargateService(props) {
        return new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
    }
}
exports.NetworkMultipleTargetGroupsFargateService = NetworkMultipleTargetGroupsFargateService;
//# sourceMappingURL=data:application/json;base64,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