"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const network_load_balanced_service_base_1 = require("../base/network-load-balanced-service-base");
/**
 * A Fargate service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkLoadBalancedFargateService extends network_load_balanced_service_base_1.NetworkLoadBalancedServiceBase {
    /**
     * Constructs a new instance of the NetworkLoadBalancedFargateService class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.assignPublicIp = props.assignPublicIp !== undefined ? props.assignPublicIp : false;
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify either a taskDefinition or an image, not both.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.FargateTaskDefinition(this, 'TaskDef', {
                memoryLimitMiB: props.memoryLimitMiB,
                cpu: props.cpu,
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            // Create log driver if logging is enabled
            const enableLogging = taskImageOptions.enableLogging !== undefined ? taskImageOptions.enableLogging : true;
            const logDriver = taskImageOptions.logDriver !== undefined
                ? taskImageOptions.logDriver : enableLogging
                ? this.createAWSLogDriver(this.node.id) : undefined;
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                logging: logDriver,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
            });
            container.addPortMappings({
                containerPort: taskImageOptions.containerPort || 80,
            });
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        this.service = new aws_ecs_1.FargateService(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            minHealthyPercent: props.minHealthyPercent,
            maxHealthyPercent: props.maxHealthyPercent,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
            platformVersion: props.platformVersion,
        });
        this.addServiceAsTarget(this.service);
    }
}
exports.NetworkLoadBalancedFargateService = NetworkLoadBalancedFargateService;
//# sourceMappingURL=data:application/json;base64,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