"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const network_load_balanced_service_base_1 = require("../base/network-load-balanced-service-base");
/**
 * An EC2 service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkLoadBalancedEc2Service extends network_load_balanced_service_base_1.NetworkLoadBalancedServiceBase {
    /**
     * Constructs a new instance of the NetworkLoadBalancedEc2Service class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify either a taskDefinition or an image, not both.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'TaskDef', {
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
                family: taskImageOptions.family,
            });
            // Create log driver if logging is enabled
            const enableLogging = taskImageOptions.enableLogging !== undefined ? taskImageOptions.enableLogging : true;
            const logDriver = taskImageOptions.logDriver !== undefined
                ? taskImageOptions.logDriver : enableLogging
                ? this.createAWSLogDriver(this.node.id) : undefined;
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                cpu: props.cpu,
                memoryLimitMiB: props.memoryLimitMiB,
                memoryReservationMiB: props.memoryReservationMiB,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
                logging: logDriver,
            });
            container.addPortMappings({
                containerPort: taskImageOptions.containerPort || 80,
            });
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: false,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            minHealthyPercent: props.minHealthyPercent,
            maxHealthyPercent: props.maxHealthyPercent,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
        });
        this.addServiceAsTarget(this.service);
    }
}
exports.NetworkLoadBalancedEc2Service = NetworkLoadBalancedEc2Service;
//# sourceMappingURL=data:application/json;base64,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