"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("../../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const aws_route53_1 = require("../../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The base class for NetworkLoadBalancedEc2Service and NetworkLoadBalancedFargateService services.
 */
class NetworkLoadBalancedServiceBase extends cdk.Construct {
    /**
     * Constructs a new instance of the NetworkLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        const internetFacing = props.publicLoadBalancer !== undefined ? props.publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer !== undefined ? props.loadBalancer :
            new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, 'LB', lbProps);
        const listenerPort = props.listenerPort !== undefined ? props.listenerPort : 80;
        const targetProps = {
            port: 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', { port: listenerPort });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            new aws_route53_1.ARecord(this, 'DNS', {
                zone: props.domainZone,
                recordName: props.domainName,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.NetworkLoadBalancer) {
            this._networkLoadBalancer = loadBalancer;
        }
        if (props.loadBalancer === undefined) {
            new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * The Network Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._networkLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._networkLoadBalancer;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.NetworkLoadBalancedServiceBase = NetworkLoadBalancedServiceBase;
//# sourceMappingURL=data:application/json;base64,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