"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            // tslint:disable-next-line: max-line-length
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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