"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
        }
    }
}
exports.UserData = UserData;
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        const shebang = this.props.shebang !== undefined ? this.props.shebang : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${params.arguments}`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : []),
        ].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        this.addCommands(`&'${params.filePath}' ${params.arguments}`, `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1kYXRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1kYXRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQ0EscUNBQXlELENBQUMsZ0RBQWdEO0FBQzFHLG1EQUFzRDtBQStDdEQ7O0dBRUc7QUFDSCxNQUFzQixRQUFRO0lBQzFCOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxVQUFnQyxFQUFFO1FBQ3JELE9BQU8sSUFBSSxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQVU7UUFDcEIsT0FBTyxJQUFJLGVBQWUsRUFBRSxDQUFDO0lBQ2pDLENBQUM7SUFDRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBZTtRQUNoQyxNQUFNLFFBQVEsR0FBRyxJQUFJLGNBQWMsRUFBRSxDQUFDO1FBQ3RDLFFBQVEsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDOUIsT0FBTyxRQUFRLENBQUM7SUFDcEIsQ0FBQztJQUNNLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxFQUF1QjtRQUNwRCxRQUFRLEVBQUUsRUFBRTtZQUNSLEtBQUssbUNBQW1CLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxRQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7WUFDM0QsS0FBSyxtQ0FBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQztTQUNsRTtJQUNMLENBQUM7Q0EyQko7QUFyREQsNEJBcURDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFNLGFBQWMsU0FBUSxRQUFRO0lBR2hDLFlBQTZCLFFBQThCLEVBQUU7UUFDekQsS0FBSyxFQUFFLENBQUM7UUFEaUIsVUFBSyxHQUFMLEtBQUssQ0FBMkI7UUFGNUMsVUFBSyxHQUFhLEVBQUUsQ0FBQztRQUNyQixnQkFBVyxHQUFhLEVBQUUsQ0FBQztJQUc1QyxDQUFDO0lBQ00sV0FBVyxDQUFDLEdBQUcsUUFBa0I7UUFDcEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ00saUJBQWlCLENBQUMsR0FBRyxRQUFrQjtRQUMxQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFDTSxNQUFNO1FBQ1QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDO1FBQ3RGLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFDTSxvQkFBb0IsQ0FBQyxNQUF5QjtRQUNqRCxNQUFNLE1BQU0sR0FBRyxRQUFRLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBVSxJQUFJLE1BQU0sQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUN0RSxNQUFNLFNBQVMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxTQUFTLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFFBQVEsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQ3RILElBQUksQ0FBQyxXQUFXLENBQUMsdUJBQXVCLFNBQVMsSUFBSSxFQUFFLGNBQWMsTUFBTSxNQUFNLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFDL0YsT0FBTyxTQUFTLENBQUM7SUFDckIsQ0FBQztJQUNNLHFCQUFxQixDQUFDLE1BQTBCO1FBQ25ELElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLGFBQWEsTUFBTSxDQUFDLFFBQVEsR0FBRyxFQUFFLElBQUksTUFBTSxDQUFDLFFBQVEsS0FBSyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUM1RyxDQUFDO0lBQ00sc0JBQXNCLENBQUMsUUFBa0I7UUFDNUMsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNqQyxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsWUFBMEIsQ0FBQyxDQUFDO1FBQ2hGLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxtQ0FBbUMsS0FBSyxDQUFDLFNBQVMsZUFBZSxVQUFVLGFBQWEsS0FBSyxDQUFDLE1BQU0sOERBQThELENBQUMsQ0FBQztJQUMvTCxDQUFDO0lBQ08saUJBQWlCO1FBQ3JCLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzdCLE9BQU8sQ0FBQyxzQkFBc0IsRUFBRSxhQUFhLEVBQUUsR0FBRyxJQUFJLENBQUMsV0FBVyxFQUFFLEdBQUcsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO1NBQ2xHO1FBQ0QsT0FBTyxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0o7QUFDRDs7R0FFRztBQUNILE1BQU0sZUFBZ0IsU0FBUSxRQUFRO0lBR2xDO1FBQ0ksS0FBSyxFQUFFLENBQUM7UUFISyxVQUFLLEdBQWEsRUFBRSxDQUFDO1FBQ3JCLGdCQUFXLEdBQWEsRUFBRSxDQUFDO0lBRzVDLENBQUM7SUFDTSxXQUFXLENBQUMsR0FBRyxRQUFrQjtRQUNwQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFDTSxpQkFBaUIsQ0FBQyxHQUFHLFFBQWtCO1FBQzFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUNNLE1BQU07UUFDVCxPQUFPLGVBQWUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7WUFDaEQsR0FBRyxJQUFJLENBQUMsS0FBSztZQUNiLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1NBQzlELENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUM7SUFDaEMsQ0FBQztJQUNNLG9CQUFvQixDQUFDLE1BQXlCO1FBQ2pELE1BQU0sU0FBUyxHQUFHLENBQUMsTUFBTSxDQUFDLFNBQVMsSUFBSSxNQUFNLENBQUMsU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsV0FBVyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDekgsSUFBSSxDQUFDLFdBQVcsQ0FBQyw0QkFBNEIsU0FBUyxXQUFXLEVBQUUsOEJBQThCLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBVSxXQUFXLE1BQU0sQ0FBQyxTQUFTLFlBQVksU0FBUyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ2hNLE9BQU8sU0FBUyxDQUFDO0lBQ3JCLENBQUM7SUFDTSxxQkFBcUIsQ0FBQyxNQUEwQjtRQUNuRCxJQUFJLENBQUMsV0FBVyxDQUFDLEtBQUssTUFBTSxDQUFDLFFBQVEsS0FBSyxNQUFNLENBQUMsU0FBUyxFQUFFLEVBQUUsdURBQXVELE1BQU0sQ0FBQyxRQUFRLHdCQUF3QixDQUFDLENBQUM7SUFDbEssQ0FBQztJQUNNLHNCQUFzQixDQUFDLFFBQWtCO1FBQzVDLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFlBQTBCLENBQUMsQ0FBQztRQUNoRixJQUFJLENBQUMsaUJBQWlCLENBQUMsc0JBQXNCLEtBQUssQ0FBQyxTQUFTLGVBQWUsVUFBVSxhQUFhLEtBQUssQ0FBQyxNQUFNLDRDQUE0QyxDQUFDLENBQUM7SUFDaEssQ0FBQztJQUNPLGlCQUFpQjtRQUNyQixJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM3QixPQUFPLENBQUMsUUFBUSxFQUFFLG9EQUFvRCxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsRUFBRSxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUM7U0FDOUc7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDSjtBQUNEOztHQUVHO0FBQ0gsTUFBTSxjQUFlLFNBQVEsUUFBUTtJQUVqQztRQUNJLEtBQUssRUFBRSxDQUFDO1FBRkssVUFBSyxHQUFhLEVBQUUsQ0FBQztJQUd0QyxDQUFDO0lBQ00sV0FBVyxDQUFDLEdBQUcsUUFBa0I7UUFDcEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBQ00saUJBQWlCO1FBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMsOEdBQThHLENBQUMsQ0FBQztJQUNwSSxDQUFDO0lBQ00sTUFBTTtRQUNULE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUNNLG9CQUFvQjtRQUN2QixNQUFNLElBQUksS0FBSyxDQUFDLGlIQUFpSCxDQUFDLENBQUM7SUFDdkksQ0FBQztJQUNNLHFCQUFxQjtRQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLGtIQUFrSCxDQUFDLENBQUM7SUFDeEksQ0FBQztJQUNNLHNCQUFzQjtRQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLG1IQUFtSCxDQUFDLENBQUM7SUFDekksQ0FBQztDQUNKIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSUJ1Y2tldCB9IGZyb20gXCIuLi8uLi9hd3MtczNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMydcbmltcG9ydCB7IENmbkVsZW1lbnQsIFJlc291cmNlLCBTdGFjayB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgT3BlcmF0aW5nU3lzdGVtVHlwZSB9IGZyb20gJy4vbWFjaGluZS1pbWFnZSc7XG4vKipcbiAqIE9wdGlvbnMgd2hlbiBjb25zdHJ1Y3RpbmcgVXNlckRhdGEgZm9yIExpbnV4XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGludXhVc2VyRGF0YU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFNoZWJhbmcgZm9yIHRoZSBVc2VyRGF0YSBzY3JpcHRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFwiIyEvYmluL2Jhc2hcIlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNoZWJhbmc/OiBzdHJpbmc7XG59XG4vKipcbiAqIE9wdGlvbnMgd2hlbiBkb3dubG9hZGluZyBmaWxlcyBmcm9tIFMzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNEb3dubG9hZE9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhlIFMzIGJ1Y2tldCB0byBkb3dubG9hZCBmcm9tXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0OiBJQnVja2V0O1xuICAgIC8qKlxuICAgICAqIFRoZSBrZXkgb2YgdGhlIGZpbGUgdG8gZG93bmxvYWRcbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXRLZXk6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgbG9jYWwgZmlsZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IExpbnV4ICAgLSAvdG1wL2J1Y2tldEtleVxuICAgICAqICAgICAgICAgIFdpbmRvd3MgLSAlVEVNUCUvYnVja2V0S2V5XG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9jYWxGaWxlPzogc3RyaW5nO1xufVxuLyoqXG4gKiBPcHRpb25zIHdoZW4gZXhlY3V0aW5nIGEgZmlsZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFeGVjdXRlRmlsZU9wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFRoZSBwYXRoIHRvIHRoZSBmaWxlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGZpbGVQYXRoOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGFyZ3VtZW50cyB0byBiZSBwYXNzZWQgdG8gdGhlIGZpbGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBhcmd1bWVudHMgYXJlIHBhc3NlZCB0byB0aGUgZmlsZS5cbiAgICAgKi9cbiAgICByZWFkb25seSBhcmd1bWVudHM/OiBzdHJpbmc7XG59XG4vKipcbiAqIEluc3RhbmNlIFVzZXIgRGF0YVxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgVXNlckRhdGEge1xuICAgIC8qKlxuICAgICAqIENyZWF0ZSBhIHVzZXJkYXRhIG9iamVjdCBmb3IgTGludXggaG9zdHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZvckxpbnV4KG9wdGlvbnM6IExpbnV4VXNlckRhdGFPcHRpb25zID0ge30pOiBVc2VyRGF0YSB7XG4gICAgICAgIHJldHVybiBuZXcgTGludXhVc2VyRGF0YShvcHRpb25zKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlIGEgdXNlcmRhdGEgb2JqZWN0IGZvciBXaW5kb3dzIGhvc3RzXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmb3JXaW5kb3dzKCk6IFVzZXJEYXRhIHtcbiAgICAgICAgcmV0dXJuIG5ldyBXaW5kb3dzVXNlckRhdGEoKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlIGEgdXNlcmRhdGEgb2JqZWN0IHdpdGggY3VzdG9tIGNvbnRlbnRcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGN1c3RvbShjb250ZW50OiBzdHJpbmcpOiBVc2VyRGF0YSB7XG4gICAgICAgIGNvbnN0IHVzZXJEYXRhID0gbmV3IEN1c3RvbVVzZXJEYXRhKCk7XG4gICAgICAgIHVzZXJEYXRhLmFkZENvbW1hbmRzKGNvbnRlbnQpO1xuICAgICAgICByZXR1cm4gdXNlckRhdGE7XG4gICAgfVxuICAgIHB1YmxpYyBzdGF0aWMgZm9yT3BlcmF0aW5nU3lzdGVtKG9zOiBPcGVyYXRpbmdTeXN0ZW1UeXBlKTogVXNlckRhdGEge1xuICAgICAgICBzd2l0Y2ggKG9zKSB7XG4gICAgICAgICAgICBjYXNlIE9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVg6IHJldHVybiBVc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgICAgICAgICAgY2FzZSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1M6IHJldHVybiBVc2VyRGF0YS5mb3JXaW5kb3dzKCk7XG4gICAgICAgIH1cbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIG9uZSBvciBtb3JlIGNvbW1hbmRzIHRvIHRoZSB1c2VyIGRhdGFcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgYWRkQ29tbWFuZHMoLi4uY29tbWFuZHM6IHN0cmluZ1tdKTogdm9pZDtcbiAgICAvKipcbiAgICAgKiBBZGQgb25lIG9yIG1vcmUgY29tbWFuZHMgdG8gdGhlIHVzZXIgZGF0YSB0aGF0IHdpbGwgcnVuIHdoZW4gdGhlIHNjcmlwdCBleGl0cy5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgYWRkT25FeGl0Q29tbWFuZHMoLi4uY29tbWFuZHM6IHN0cmluZ1tdKTogdm9pZDtcbiAgICAvKipcbiAgICAgKiBSZW5kZXIgdGhlIFVzZXJEYXRhIGZvciB1c2UgaW4gYSBjb25zdHJ1Y3RcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVuZGVyKCk6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBZGRzIGNvbW1hbmRzIHRvIGRvd25sb2FkIGEgZmlsZSBmcm9tIFMzXG4gICAgICpcbiAgICAgKiBAcmV0dXJuczogVGhlIGxvY2FsIHBhdGggdGhhdCB0aGUgZmlsZSB3aWxsIGJlIGRvd25sb2FkZWQgdG9cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgYWRkUzNEb3dubG9hZENvbW1hbmQocGFyYW1zOiBTM0Rvd25sb2FkT3B0aW9ucyk6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBBZGRzIGNvbW1hbmRzIHRvIGV4ZWN1dGUgYSBmaWxlXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IGFkZEV4ZWN1dGVGaWxlQ29tbWFuZChwYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQ7XG4gICAgLyoqXG4gICAgICogQWRkcyBhIGNvbW1hbmQgd2hpY2ggd2lsbCBzZW5kIGEgY2ZuLXNpZ25hbCB3aGVuIHRoZSB1c2VyIGRhdGEgc2NyaXB0IGVuZHNcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgYWRkU2lnbmFsT25FeGl0Q29tbWFuZChyZXNvdXJjZTogUmVzb3VyY2UpOiB2b2lkO1xufVxuLyoqXG4gKiBMaW51eCBJbnN0YW5jZSBVc2VyIERhdGFcbiAqL1xuY2xhc3MgTGludXhVc2VyRGF0YSBleHRlbmRzIFVzZXJEYXRhIHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxpbmVzOiBzdHJpbmdbXSA9IFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgb25FeGl0TGluZXM6IHN0cmluZ1tdID0gW107XG4gICAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogTGludXhVc2VyRGF0YU9wdGlvbnMgPSB7fSkge1xuICAgICAgICBzdXBlcigpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkQ29tbWFuZHMoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHRoaXMubGluZXMucHVzaCguLi5jb21tYW5kcyk7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRPbkV4aXRDb21tYW5kcyguLi5jb21tYW5kczogc3RyaW5nW10pIHtcbiAgICAgICAgdGhpcy5vbkV4aXRMaW5lcy5wdXNoKC4uLmNvbW1hbmRzKTtcbiAgICB9XG4gICAgcHVibGljIHJlbmRlcigpOiBzdHJpbmcge1xuICAgICAgICBjb25zdCBzaGViYW5nID0gdGhpcy5wcm9wcy5zaGViYW5nICE9PSB1bmRlZmluZWQgPyB0aGlzLnByb3BzLnNoZWJhbmcgOiAnIyEvYmluL2Jhc2gnO1xuICAgICAgICByZXR1cm4gW3NoZWJhbmcsIC4uLih0aGlzLnJlbmRlck9uRXhpdExpbmVzKCkpLCAuLi50aGlzLmxpbmVzXS5qb2luKCdcXG4nKTtcbiAgICB9XG4gICAgcHVibGljIGFkZFMzRG93bmxvYWRDb21tYW5kKHBhcmFtczogUzNEb3dubG9hZE9wdGlvbnMpOiBzdHJpbmcge1xuICAgICAgICBjb25zdCBzM1BhdGggPSBgczM6Ly8ke3BhcmFtcy5idWNrZXQuYnVja2V0TmFtZX0vJHtwYXJhbXMuYnVja2V0S2V5fWA7XG4gICAgICAgIGNvbnN0IGxvY2FsUGF0aCA9IChwYXJhbXMubG9jYWxGaWxlICYmIHBhcmFtcy5sb2NhbEZpbGUubGVuZ3RoICE9PSAwKSA/IHBhcmFtcy5sb2NhbEZpbGUgOiBgL3RtcC8ke3BhcmFtcy5idWNrZXRLZXl9YDtcbiAgICAgICAgdGhpcy5hZGRDb21tYW5kcyhgbWtkaXIgLXAgJChkaXJuYW1lICcke2xvY2FsUGF0aH0nKWAsIGBhd3MgczMgY3AgJyR7czNQYXRofScgJyR7bG9jYWxQYXRofSdgKTtcbiAgICAgICAgcmV0dXJuIGxvY2FsUGF0aDtcbiAgICB9XG4gICAgcHVibGljIGFkZEV4ZWN1dGVGaWxlQ29tbWFuZChwYXJhbXM6IEV4ZWN1dGVGaWxlT3B0aW9ucyk6IHZvaWQge1xuICAgICAgICB0aGlzLmFkZENvbW1hbmRzKCdzZXQgLWUnLCBgY2htb2QgK3ggJyR7cGFyYW1zLmZpbGVQYXRofSdgLCBgJyR7cGFyYW1zLmZpbGVQYXRofScgJHtwYXJhbXMuYXJndW1lbnRzfWApO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkU2lnbmFsT25FeGl0Q29tbWFuZChyZXNvdXJjZTogUmVzb3VyY2UpOiB2b2lkIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihyZXNvdXJjZSk7XG4gICAgICAgIGNvbnN0IHJlc291cmNlSUQgPSBzdGFjay5nZXRMb2dpY2FsSWQocmVzb3VyY2Uubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG4gICAgICAgIHRoaXMuYWRkT25FeGl0Q29tbWFuZHMoYC9vcHQvYXdzL2Jpbi9jZm4tc2lnbmFsIC0tc3RhY2sgJHtzdGFjay5zdGFja05hbWV9IC0tcmVzb3VyY2UgJHtyZXNvdXJjZUlEfSAtLXJlZ2lvbiAke3N0YWNrLnJlZ2lvbn0gLWUgJGV4aXRDb2RlIHx8IGVjaG8gJ0ZhaWxlZCB0byBzZW5kIENsb3VkZm9ybWF0aW9uIFNpZ25hbCdgKTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJPbkV4aXRMaW5lcygpOiBzdHJpbmdbXSB7XG4gICAgICAgIGlmICh0aGlzLm9uRXhpdExpbmVzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIHJldHVybiBbJ2Z1bmN0aW9uIGV4aXRUcmFwKCl7JywgJ2V4aXRDb2RlPSQ/JywgLi4udGhpcy5vbkV4aXRMaW5lcywgJ30nLCAndHJhcCBleGl0VHJhcCBFWElUJ107XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIFtdO1xuICAgIH1cbn1cbi8qKlxuICogV2luZG93cyBJbnN0YW5jZSBVc2VyIERhdGFcbiAqL1xuY2xhc3MgV2luZG93c1VzZXJEYXRhIGV4dGVuZHMgVXNlckRhdGEge1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbGluZXM6IHN0cmluZ1tdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBvbkV4aXRMaW5lczogc3RyaW5nW10gPSBbXTtcbiAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgcHVibGljIGFkZENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgICAgICB0aGlzLmxpbmVzLnB1c2goLi4uY29tbWFuZHMpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkT25FeGl0Q29tbWFuZHMoLi4uY29tbWFuZHM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHRoaXMub25FeGl0TGluZXMucHVzaCguLi5jb21tYW5kcyk7XG4gICAgfVxuICAgIHB1YmxpYyByZW5kZXIoKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIGA8cG93ZXJzaGVsbD4ke1suLi4odGhpcy5yZW5kZXJPbkV4aXRMaW5lcygpKSxcbiAgICAgICAgICAgIC4uLnRoaXMubGluZXMsXG4gICAgICAgICAgICAuLi4odGhpcy5vbkV4aXRMaW5lcy5sZW5ndGggPiAwID8gWyd0aHJvdyBcIlN1Y2Nlc3NcIiddIDogW10pLFxuICAgICAgICBdLmpvaW4oJ1xcbicpfTwvcG93ZXJzaGVsbD5gO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkUzNEb3dubG9hZENvbW1hbmQocGFyYW1zOiBTM0Rvd25sb2FkT3B0aW9ucyk6IHN0cmluZyB7XG4gICAgICAgIGNvbnN0IGxvY2FsUGF0aCA9IChwYXJhbXMubG9jYWxGaWxlICYmIHBhcmFtcy5sb2NhbEZpbGUubGVuZ3RoICE9PSAwKSA/IHBhcmFtcy5sb2NhbEZpbGUgOiBgQzovdGVtcC8ke3BhcmFtcy5idWNrZXRLZXl9YDtcbiAgICAgICAgdGhpcy5hZGRDb21tYW5kcyhgbWtkaXIgKFNwbGl0LVBhdGggLVBhdGggJyR7bG9jYWxQYXRofScgKSAtZWEgMGAsIGBSZWFkLVMzT2JqZWN0IC1CdWNrZXROYW1lICcke3BhcmFtcy5idWNrZXQuYnVja2V0TmFtZX0nIC1rZXkgJyR7cGFyYW1zLmJ1Y2tldEtleX0nIC1maWxlICcke2xvY2FsUGF0aH0nIC1FcnJvckFjdGlvbiBTdG9wYCk7XG4gICAgICAgIHJldHVybiBsb2NhbFBhdGg7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRFeGVjdXRlRmlsZUNvbW1hbmQocGFyYW1zOiBFeGVjdXRlRmlsZU9wdGlvbnMpOiB2b2lkIHtcbiAgICAgICAgdGhpcy5hZGRDb21tYW5kcyhgJicke3BhcmFtcy5maWxlUGF0aH0nICR7cGFyYW1zLmFyZ3VtZW50c31gLCBgaWYgKCEkPykgeyBXcml0ZS1FcnJvciAnRmFpbGVkIHRvIGV4ZWN1dGUgdGhlIGZpbGUgXCIke3BhcmFtcy5maWxlUGF0aH1cIicgLUVycm9yQWN0aW9uIFN0b3AgfWApO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkU2lnbmFsT25FeGl0Q29tbWFuZChyZXNvdXJjZTogUmVzb3VyY2UpOiB2b2lkIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihyZXNvdXJjZSk7XG4gICAgICAgIGNvbnN0IHJlc291cmNlSUQgPSBzdGFjay5nZXRMb2dpY2FsSWQocmVzb3VyY2Uubm9kZS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuRWxlbWVudCk7XG4gICAgICAgIHRoaXMuYWRkT25FeGl0Q29tbWFuZHMoYGNmbi1zaWduYWwgLS1zdGFjayAke3N0YWNrLnN0YWNrTmFtZX0gLS1yZXNvdXJjZSAke3Jlc291cmNlSUR9IC0tcmVnaW9uICR7c3RhY2sucmVnaW9ufSAtLXN1Y2Nlc3MgKCRzdWNjZXNzLlRvU3RyaW5nKCkuVG9Mb3dlcigpKWApO1xuICAgIH1cbiAgICBwcml2YXRlIHJlbmRlck9uRXhpdExpbmVzKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgaWYgKHRoaXMub25FeGl0TGluZXMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgcmV0dXJuIFsndHJhcCB7JywgJyRzdWNjZXNzPSgkUFNJdGVtLkV4Y2VwdGlvbi5NZXNzYWdlIC1lcSBcIlN1Y2Nlc3NcIiknLCAuLi50aGlzLm9uRXhpdExpbmVzLCAnYnJlYWsnLCAnfSddO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG59XG4vKipcbiAqIEN1c3RvbSBJbnN0YW5jZSBVc2VyIERhdGFcbiAqL1xuY2xhc3MgQ3VzdG9tVXNlckRhdGEgZXh0ZW5kcyBVc2VyRGF0YSB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBsaW5lczogc3RyaW5nW10gPSBbXTtcbiAgICBjb25zdHJ1Y3RvcigpIHtcbiAgICAgICAgc3VwZXIoKTtcbiAgICB9XG4gICAgcHVibGljIGFkZENvbW1hbmRzKC4uLmNvbW1hbmRzOiBzdHJpbmdbXSkge1xuICAgICAgICB0aGlzLmxpbmVzLnB1c2goLi4uY29tbWFuZHMpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkT25FeGl0Q29tbWFuZHMoKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ3VzdG9tVXNlckRhdGEgZG9lcyBub3Qgc3VwcG9ydCBhZGRPbkV4aXRDb21tYW5kcywgdXNlIFVzZXJEYXRhLmZvckxpbnV4KCkgb3IgVXNlckRhdGEuZm9yV2luZG93cygpIGluc3RlYWQuJyk7XG4gICAgfVxuICAgIHB1YmxpYyByZW5kZXIoKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIHRoaXMubGluZXMuam9pbignXFxuJyk7XG4gICAgfVxuICAgIHB1YmxpYyBhZGRTM0Rvd25sb2FkQ29tbWFuZCgpOiBzdHJpbmcge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0N1c3RvbVVzZXJEYXRhIGRvZXMgbm90IHN1cHBvcnQgYWRkUzNEb3dubG9hZENvbW1hbmQsIHVzZSBVc2VyRGF0YS5mb3JMaW51eCgpIG9yIFVzZXJEYXRhLmZvcldpbmRvd3MoKSBpbnN0ZWFkLicpO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkRXhlY3V0ZUZpbGVDb21tYW5kKCk6IHZvaWQge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0N1c3RvbVVzZXJEYXRhIGRvZXMgbm90IHN1cHBvcnQgYWRkRXhlY3V0ZUZpbGVDb21tYW5kLCB1c2UgVXNlckRhdGEuZm9yTGludXgoKSBvciBVc2VyRGF0YS5mb3JXaW5kb3dzKCkgaW5zdGVhZC4nKTtcbiAgICB9XG4gICAgcHVibGljIGFkZFNpZ25hbE9uRXhpdENvbW1hbmQoKTogdm9pZCB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ3VzdG9tVXNlckRhdGEgZG9lcyBub3Qgc3VwcG9ydCBhZGRTaWduYWxPbkV4aXRDb21tYW5kLCB1c2UgVXNlckRhdGEuZm9yTGludXgoKSBvciBVc2VyRGF0YS5mb3JXaW5kb3dzKCkgaW5zdGVhZC4nKTtcbiAgICB9XG59XG4iXX0=