"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Protocol for use in Connection Rules
 */
var Protocol;
(function (Protocol) {
    Protocol["ALL"] = "-1";
    Protocol["TCP"] = "tcp";
    Protocol["UDP"] = "udp";
    Protocol["ICMP"] = "icmp";
    Protocol["ICMPV6"] = "58";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
class Port {
    constructor(props) {
        this.props = props;
        this.canInlineRule = !core_1.Token.isUnresolved(props.fromPort) && !core_1.Token.isUnresolved(props.toPort);
    }
    /**
     * A single TCP port
     */
    static tcp(port) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: port,
            toPort: port,
            stringRepresentation: renderPort(port),
        });
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new Port({
            protocol: Protocol.TCP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'ALL PORTS',
        });
    }
    /**
     * A single UDP port
     */
    static udp(port) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: port,
            toPort: port,
            stringRepresentation: `UDP ${renderPort(port)}`,
        });
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: startPort,
            toPort: endPort,
            stringRepresentation: `UDP ${renderPort(startPort)}-${renderPort(endPort)}`,
        });
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new Port({
            protocol: Protocol.UDP,
            fromPort: 0,
            toPort: 65535,
            stringRepresentation: 'UDP ALL PORTS',
        });
    }
    /**
     * A specific combination of ICMP type and code
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     */
    static icmpTypeAndCode(type, code) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: code,
            stringRepresentation: `ICMP Type ${type} Code ${code}`,
        });
    }
    /**
     * All codes for a single ICMP type
     */
    static icmpType(type) {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: type,
            toPort: -1,
            stringRepresentation: `ICMP Type ${type}`,
        });
    }
    /**
     * ICMP ping (echo) traffic
     */
    static icmpPing() {
        return Port.icmpType(8);
    }
    /**
     * All ICMP traffic
     */
    static allIcmp() {
        return new Port({
            protocol: Protocol.ICMP,
            fromPort: -1,
            toPort: -1,
            stringRepresentation: 'ALL ICMP',
        });
    }
    /**
     * All traffic
     */
    static allTraffic() {
        return new Port({
            protocol: Protocol.ALL,
            stringRepresentation: 'ALL TRAFFIC',
        });
    }
    /**
     * Produce the ingress/egress rule JSON for the given connection
     */
    toRuleJson() {
        return {
            ipProtocol: this.props.protocol,
            fromPort: this.props.fromPort,
            toPort: this.props.toPort,
        };
    }
    toString() {
        return this.props.stringRepresentation;
    }
}
exports.Port = Port;
function renderPort(port) {
    return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
}
//# sourceMappingURL=data:application/json;base64,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