"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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