"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            'x86_64',
            this.storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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