"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Connection endpoint of a database cluster or instance
 *
 * Consists of a combination of hostname and port.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param address - The hostname or address of the endpoint
     * @param port - The port number of the endpoint
     */
    constructor(address, port) {
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.port = port;
        const portDesc = core_1.Token.isUnresolved(port) ? core_1.Token.asString(port) : port;
        this.socketAddress = `${address}:${portDesc}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.port)) {
            return core_1.Token.asString(this.port);
        }
        else {
            return this.port.toString();
        }
    }
}
exports.Endpoint = Endpoint;
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
//# sourceMappingURL=data:application/json;base64,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