"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * Defines a CloudWatch event rule which triggers for rule events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule compliance events.
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * Imports an existing rule.
     *
     * @param configRuleName the name of the rule
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Restrict scope of changes to a specific resource.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param type the resource type
     * @param identifier the resource identifier
     */
    scopeToResource(type, identifier) {
        this.scopeTo({
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        });
    }
    /**
     * Restrict scope of changes to specific resource types.
     *
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     *
     * @param types resource types
     */
    scopeToResources(...types) {
        this.scopeTo({
            complianceResourceTypes: types,
        });
    }
    /**
     * Restrict scope of changes to a specific tag.
     *
     * @param key the tag key
     * @param value the tag value
     */
    scopeToTag(key, value) {
        this.scopeTo({
            tagKey: key,
            tagValue: value,
        });
    }
    scopeTo(scope) {
        if (!this.isManaged && !this.isCustomWithChanges) {
            throw new Error('Cannot scope rule when `configurationChanges` is set to false.');
        }
        this.scope = scope;
    }
}
/**
 * The maximum frequency at which the AWS Config rule runs evaluations.
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    /**
     * 1 hour.
     */
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    /**
     * 3 hours.
     */
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    /**
     * 6 hours.
     */
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    /**
     * 12 hours.
     */
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    /**
     * 24 hours.
     */
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * A new managed rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * A new custom rule.
 *
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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