"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_resources_1 = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain
 */
class UserPoolDomain extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_a = props.cognitoDomain) === null || _a === void 0 ? void 0 : _a.domainPrefix) && !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) || ((_c = props.customDomain) === null || _c === void 0 ? void 0 : _c.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     */
    get cloudFrontDomainName() {
        const sdkCall = {
            service: 'CognitoIdentityServiceProvider',
            action: 'describeUserPoolDomain',
            parameters: {
                Domain: this.domainName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
        };
        const customResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
            resourceType: 'Custom::UserPoolCloudFrontDomainName',
            onCreate: sdkCall,
            onUpdate: sdkCall,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                // DescribeUserPoolDomain only supports access level '*'
                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                resources: ['*'],
            }),
        });
        return customResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool
     * @param options options to customize the behaviour of this method.
     */
    signInUrl(client, options) {
        var _a;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_a = options.signInPath) !== null && _a !== void 0 ? _a : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
//# sourceMappingURL=data:application/json;base64,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