"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: this.configureOAuthFlows(),
            allowedOAuthScopes: this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 ? callbackUrls : undefined,
            allowedOAuthFlowsUserPoolClient: props.oAuth ? true : undefined,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        var _a, _b, _c, _d, _e;
        const authFlows = [];
        if ((_a = props.authFlows) === null || _a === void 0 ? void 0 : _a.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if ((_b = props.authFlows) === null || _b === void 0 ? void 0 : _b.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if ((_c = props.authFlows) === null || _c === void 0 ? void 0 : _c.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if ((_d = props.authFlows) === null || _d === void 0 ? void 0 : _d.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        if ((_e = props.authFlows) === null || _e === void 0 ? void 0 : _e.refreshToken) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,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