"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.stringValue({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.stringValue({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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