"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = stage.pipeline.node.uniqueId + 'SourceEventRule' + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(id)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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