"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * CodePipeline invoke Action that is provided by an AWS Lambda function.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html
 */
class LambdaInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.lambda,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 5,
                minOutputs: 0,
                maxOutputs: 5,
            },
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the Lambda function this action points to.
     * Variables in Lambda invoke actions are defined by calling the PutJobSuccessResult CodePipeline API call
     * with the 'outputVariables' property filled.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'outputVariables' section of the PutJobSuccessResult
     *   request that the Lambda function calls when the action is invoked
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // allow pipeline to list functions
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [this.props.lambda.functionArn],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        this.props.lambda.addToRolePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult'],
        }));
        return {
            configuration: {
                FunctionName: this.props.lambda.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString(this.props.userParameters),
            },
        };
    }
}
exports.LambdaInvokeAction = LambdaInvokeAction;
//# sourceMappingURL=data:application/json;base64,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